<?php
/**
 * رفع درجات الفصل الدراسي من ملف Excel/CSV
 * Upload Semester Grades from Excel/CSV File
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';
require_once __DIR__ . '/classes/ExcelReader.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasPermission('enter_grades') && !$auth->hasPermission('manage_grades') && !$auth->hasRole('admin'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();

$college_id = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$semester_number = isset($_GET['semester_number']) ? (int)$_GET['semester_number'] : 0;
$academic_year = isset($_GET['academic_year']) ? trim($_GET['academic_year']) : '';
$semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;

$message = '';
$message_type = '';

// جلب الكليات المتاحة
$available_colleges = [];
if ($auth->hasRole('admin')) {
    $available_colleges = $db->fetchAll("SELECT id, name_ar FROM colleges ORDER BY name_ar");
} else {
    $user = $auth->getCurrentUser();
    if ($user && isset($user['college_id'])) {
        $college = $db->fetchOne("SELECT id, name_ar FROM colleges WHERE id = ?", [$user['college_id']]);
        if ($college) {
            $available_colleges = [$college];
            if ($college_id == 0) {
                $college_id = $college['id'];
            }
        }
    }
}

// جلب التخصصات المتاحة
$available_majors = [];
if ($college_id > 0) {
    if ($auth->hasRole('admin') || ($auth->belongsToCollege($college_id))) {
        $available_majors = $db->fetchAll(
            "SELECT id, name_ar, code FROM majors WHERE college_id = ? ORDER BY name_ar",
            [$college_id]
        );
    }
}

// جلب أرقام الفصول المتاحة
$available_semester_numbers = [];
if ($major_id > 0) {
    $available_semester_numbers = $db->fetchAll(
        "SELECT DISTINCT semester_number 
         FROM semesters 
         WHERE major_id = ? 
         ORDER BY semester_number",
        [$major_id]
    );
    $available_semester_numbers = array_column($available_semester_numbers, 'semester_number');
}

// جلب السنوات الدراسية المتاحة
$available_academic_years = [];
if ($major_id > 0 && $semester_number > 0) {
    $available_academic_years = $db->fetchAll(
        "SELECT DISTINCT academic_year 
         FROM semesters 
         WHERE major_id = ? AND semester_number = ? 
         ORDER BY academic_year DESC",
        [$major_id, $semester_number]
    );
    $available_academic_years = array_column($available_academic_years, 'academic_year');
}

// تحديد semester_id
if ($semester_id <= 0 && $major_id > 0 && $semester_number > 0 && $academic_year !== '') {
    $semester = $db->fetchOne(
        "SELECT id FROM semesters WHERE major_id = ? AND semester_number = ? AND academic_year = ?",
        [$major_id, $semester_number, $academic_year]
    );
    if ($semester) {
        $semester_id = $semester['id'];
    }
}

// معالجة رفع الملف
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['grades_file']) && isset($_POST['upload_grades'])) {
    $post_college_id = isset($_POST['college_id']) ? (int)$_POST['college_id'] : 0;
    $post_major_id = isset($_POST['major_id']) ? (int)$_POST['major_id'] : 0;
    $post_semester_number = isset($_POST['semester_number']) ? (int)$_POST['semester_number'] : 0;
    $post_academic_year = isset($_POST['academic_year']) ? trim($_POST['academic_year']) : '';
    
    // تحديد semester_id
    $post_semester_id = 0;
    if ($post_major_id > 0 && $post_semester_number > 0 && $post_academic_year !== '') {
        $semester_check = $db->fetchOne(
            "SELECT id FROM semesters WHERE major_id = ? AND semester_number = ? AND academic_year = ?",
            [$post_major_id, $post_semester_number, $post_academic_year]
        );
        if ($semester_check) {
            $post_semester_id = $semester_check['id'];
        }
    }
    
    if ($post_semester_id <= 0) {
        $message = 'خطأ: يرجى تحديد الفصل الدراسي بشكل صحيح';
        $message_type = 'error';
    } elseif (!isset($_FILES['grades_file']) || $_FILES['grades_file']['error'] !== UPLOAD_ERR_OK) {
        $message = 'خطأ: لم يتم رفع الملف بنجاح';
        $message_type = 'error';
    } else {
        $file = $_FILES['grades_file'];
        $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($file_extension, ['csv', 'xlsx', 'xls'])) {
            $message = 'خطأ: نوع الملف غير مدعوم. يرجى رفع ملف CSV أو Excel';
            $message_type = 'error';
        } else {
            // قراءة الملف مباشرة (CSV) أو باستخدام ExcelReader
            $file_data = [];
            $headers = [];
            
            try {
                if ($file_extension === 'csv') {
                    // قراءة ملف CSV مباشرة مع معالجة الترميز
                    $file_content = @file_get_contents($file['tmp_name']);
                    
                    if ($file_content === false || empty($file_content)) {
                        throw new Exception('فشل قراءة الملف أو الملف فارغ');
                    }
                    
                    // إزالة BOM إذا كان موجوداً
                    if (substr($file_content, 0, 3) === "\xEF\xBB\xBF") {
                        $file_content = substr($file_content, 3);
                    }
                    
                    // التحقق من الترميز وتحويله
                    $detected_encoding = mb_detect_encoding($file_content, ['UTF-8', 'Windows-1256', 'ISO-8859-1', 'ISO-8859-6', 'ASCII'], true);
                    if ($detected_encoding && $detected_encoding !== 'UTF-8') {
                        $file_content = mb_convert_encoding($file_content, 'UTF-8', $detected_encoding);
                    } elseif (!mb_check_encoding($file_content, 'UTF-8')) {
                        $file_content = mb_convert_encoding($file_content, 'UTF-8', 'auto');
                    }
                    
                    // حفظ المحتوى في ملف مؤقت لاستخدام fgetcsv
                    $temp_file = tempnam(sys_get_temp_dir(), 'csv_');
                    if ($temp_file === false || file_put_contents($temp_file, $file_content) === false) {
                        throw new Exception('فشل إنشاء ملف مؤقت');
                    }
                    
                    $handle = @fopen($temp_file, 'r');
                    if ($handle === false) {
                        @unlink($temp_file);
                        throw new Exception('فشل فتح الملف للقراءة');
                    }
                    
                    // قراءة العناوين من السطر الأول
                    $headers = @fgetcsv($handle, 0, ',', '"', '"');
                    if ($headers === false || empty($headers)) {
                        fclose($handle);
                        @unlink($temp_file);
                        throw new Exception('فشل قراءة العناوين من الملف');
                    }
                    
                    // تنظيف العناوين
                    foreach ($headers as &$header) {
                        $header = trim($header);
                        // إزالة BOM من كل عنوان بشكل منفصل
                        if (substr($header, 0, 3) === "\xEF\xBB\xBF") {
                            $header = substr($header, 3);
                        }
                        if (!mb_check_encoding($header, 'UTF-8')) {
                            $header = mb_convert_encoding($header, 'UTF-8', 'auto');
                        }
                        // إزالة أي رموز غير مرئية
                        $header = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $header);
                        $header = trim($header);
                    }
                    unset($header); // إزالة المرجع
                    
                    // التحقق من أن هناك عناوين صحيحة
                    $headers = array_filter($headers, function($h) { return !empty($h); });
                    $headers = array_values($headers); // إعادة ترقيم المفاتيح
                    
                    if (empty($headers) || count($headers) < 2) {
                        fclose($handle);
                        @unlink($temp_file);
                        throw new Exception('العناوين غير صحيحة. يجب أن يكون هناك عمود للرقم الجامعي وعدة أعمدة للمواد. العناوين الموجودة: ' . implode(', ', $headers));
                    }
                    
                    // قراءة البيانات من السطور التالية
                    $row_num = 1;
                    while (($row = @fgetcsv($handle, 0, ',', '"', '"')) !== false) {
                        $row_num++;
                        
                        // تخطي الصفوف الفارغة تماماً
                        $filtered_row = array_filter($row, function($cell) {
                            return !empty(trim((string)$cell));
                        });
                        
                        if (empty($filtered_row)) {
                            continue;
                        }
                        
                        // تحويل encoding لكل خلية
                        $clean_row = [];
                        for ($i = 0; $i < count($headers) && $i < count($row); $i++) {
                            $cell = isset($row[$i]) ? (string)trim($row[$i]) : '';
                            if (!empty($cell)) {
                                if (!mb_check_encoding($cell, 'UTF-8')) {
                                    $cell = mb_convert_encoding($cell, 'UTF-8', 'auto');
                                }
                                // إزالة أي رموز غير مرئية (لكن نترك الأحرف العربية)
                                $cell = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $cell);
                                $cell = trim($cell);
                            }
                            $clean_row[] = $cell;
                        }
                        
                        // التأكد من أن الصف يحتوي على رقم جامعي على الأقل
                        if (!empty($clean_row) && !empty($clean_row[0])) {
                            $file_data[] = $clean_row;
                        }
                    }
                    
                    fclose($handle);
                    @unlink($temp_file);
                    
                    // التحقق من وجود بيانات
                    if (empty($file_data)) {
                        throw new Exception('لا توجد بيانات صحيحة في الملف. يرجى التحقق من أن الملف يحتوي على صفوف بيانات بعد العناوين');
                    }
                } else {
                    // للـ Excel، استخدم ExcelReader
                    $message = 'ملاحظة: يرجى تحويل ملف Excel إلى CSV ثم رفعه. سيتم إضافة دعم Excel قريباً.';
                    $message_type = 'warning';
                }
            } catch (Exception $e) {
                $message = 'خطأ في قراءة الملف: ' . htmlspecialchars($e->getMessage());
                $message_type = 'error';
            }
            
            if (!empty($file_data) && !empty($headers) && count($headers) > 1) {
                // معالجة البيانات
                $first_column = trim($headers[0]);
                $course_codes = [];
                for ($i = 1; $i < count($headers); $i++) {
                    $course_codes[] = trim($headers[$i]);
                }
                
                $success_count = 0;
                $error_count = 0;
                $errors = [];
                
                try {
                    $db->getConnection()->beginTransaction();
                    
                    // جلب معلومات الفصل الدراسي
                    $semester_info = $db->fetchOne(
                        "SELECT * FROM semesters WHERE id = ?",
                        [$post_semester_id]
                    );
                    
                    if (!$semester_info) {
                        throw new Exception('الفصل الدراسي غير موجود');
                    }
                    
                    // جلب جميع المواد في الفصل مع أكوادها
                    $courses_map = [];
                    $courses = $db->fetchAll(
                        "SELECT id, code, name_ar FROM courses WHERE major_id = ? AND semester_id = ?",
                        [$post_major_id, $post_semester_id]
                    );
                    
                    foreach ($courses as $course) {
                        $courses_map[$course['code']] = $course;
                    }
                    
                    // معالجة كل صف في الملف
                    foreach ($file_data as $row_index => $row) {
                        if (empty($row) || !isset($row[0]) || empty(trim($row[0]))) {
                            continue; // تخطي الصفوف الفارغة
                        }
                        
                        $student_number = trim($row[0]);
                        
                        // البحث عن الطالب
                        $student = $db->fetchOne(
                            "SELECT id, student_number FROM students WHERE student_number = ? AND major_id = ?",
                            [$student_number, $post_major_id]
                        );
                        
                        if (!$student) {
                            $error_count++;
                            $errors[] = "السطر " . ($row_index + 2) . ": الطالب برقم $student_number غير موجود";
                            continue;
                        }
                        
                        $student_id = $student['id'];
                        
                        // معالجة درجات كل مادة
                        for ($col_index = 1; $col_index < count($row) && ($col_index - 1) < count($course_codes); $col_index++) {
                            $course_code = trim($course_codes[$col_index - 1] ?? '');
                            if (empty($course_code)) {
                                continue;
                            }
                            
                            // التحقق من وجود المادة
                            if (!isset($courses_map[$course_code])) {
                                continue;
                            }
                            
                            $course = $courses_map[$course_code];
                            $course_id = $course['id'];
                            
                            // الحصول على الدرجة
                            $grade_value = isset($row[$col_index]) ? trim($row[$col_index]) : '';
                            
                            if (empty($grade_value)) {
                                continue; // تخطي الدرجات الفارغة
                            }
                            
                            // التحقق من تسجيل الطالب في المادة
                            $student_course = $db->fetchOne(
                                "SELECT id FROM student_courses 
                                 WHERE student_id = ? AND course_id = ? AND semester_id = ?",
                                [$student_id, $course_id, $post_semester_id]
                            );
                            
                            if (!$student_course) {
                                // تسجيل الطالب في المادة إذا لم يكن مسجلاً
                                try {
                                    $db->query(
                                        "INSERT INTO student_courses (student_id, course_id, semester_id, registration_date) 
                                         VALUES (?, ?, ?, CURDATE())",
                                        [$student_id, $course_id, $post_semester_id]
                                    );
                                    $student_course = $db->fetchOne(
                                        "SELECT id FROM student_courses 
                                         WHERE student_id = ? AND course_id = ? AND semester_id = ?",
                                        [$student_id, $course_id, $post_semester_id]
                                    );
                                } catch (Exception $e) {
                                    $error_count++;
                                    $errors[] = "السطر " . ($row_index + 2) . ": فشل تسجيل الطالب $student_number في المادة $course_code";
                                    continue;
                                }
                            }
                            
                            $student_course_id = $student_course['id'];
                            
                            // معالجة الدرجة
                            $grade_data = $gradeCalculator->processGradeInput($grade_value);
                            
                            if (!$grade_data) {
                                $error_count++;
                                continue;
                            }
                            
                            // تحديد قيمة marks (إذا كانت نصية، نحفظ النص الأصلي)
                            if ($grade_data['is_text']) {
                                $marks = $grade_value; // حفظ القيمة النصية الأصلية
                            } else {
                                $marks = $grade_data['marks'] !== null ? $grade_data['marks'] : 0;
                            }
                            $letter_grade = $grade_data['letter_grade'];
                            $points = $grade_data['points'];
                            
                            // حفظ أو تحديث الدرجة
                            $existing_grade = $db->fetchOne(
                                "SELECT id FROM grades WHERE student_course_id = ?",
                                [$student_course_id]
                            );
                            
                            if ($existing_grade) {
                                // تحديث الدرجة الموجودة
                                $db->query(
                                    "UPDATE grades SET 
                                     marks = ?, 
                                     letter_grade = ?, 
                                     points = ?,
                                     entered_by = ?,
                                     updated_at = CURRENT_TIMESTAMP
                                     WHERE student_course_id = ?",
                                    [
                                        $marks,
                                        $letter_grade,
                                        $points,
                                        $_SESSION['user_id'],
                                        $student_course_id
                                    ]
                                );
                            } else {
                                // إضافة درجة جديدة
                                $db->query(
                                    "INSERT INTO grades 
                                     (student_course_id, student_id, course_id, semester_id, marks, letter_grade, points, entered_by) 
                                     VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
                                    [
                                        $student_course_id,
                                        $student_id,
                                        $course_id,
                                        $post_semester_id,
                                        $marks,
                                        $letter_grade,
                                        $points,
                                        $_SESSION['user_id']
                                    ]
                                );
                            }
                            
                            $success_count++;
                        }
                    }
                    
                    // تحديث المعدلات للطلاب
                    $updated_students = [];
                    foreach ($file_data as $row) {
                        if (empty($row) || !isset($row[0])) {
                            continue;
                        }
                        $student_number = trim($row[0]);
                        $student = $db->fetchOne(
                            "SELECT id FROM students WHERE student_number = ? AND major_id = ?",
                            [$student_number, $post_major_id]
                        );
                        if ($student && !in_array($student['id'], $updated_students)) {
                            $updated_students[] = $student['id'];
                            try {
                                $gradeCalculator->updateAllGPAs($student['id']);
                            } catch (Exception $e) {
                                error_log("Error updating GPA for student {$student['id']}: " . $e->getMessage());
                            }
                        }
                    }
                    
                    $db->getConnection()->commit();
                    
                    // بناء رسالة النجاح
                    if ($success_count > 0) {
                        $message = "تم رفع " . $success_count . " درجة بنجاح";
                        $message_type = 'success';
                    }
                    
                    if ($error_count > 0) {
                        if ($message) {
                            $message .= '<br>';
                        }
                        $message .= "عدد الأخطاء: " . $error_count;
                        if (count($errors) > 0 && count($errors) <= 10) {
                            $message .= '<br><small>' . implode('<br>', $errors) . '</small>';
                        }
                        if ($message_type !== 'success') {
                            $message_type = 'warning';
                        }
                    }
                    
                    // إعادة التوجيه
                    header('Location: upload_grades_excel.php?college_id=' . $post_college_id . 
                           '&major_id=' . $post_major_id . 
                           '&semester_number=' . $post_semester_number . 
                           '&academic_year=' . urlencode($post_academic_year) . 
                           '&msg=' . urlencode($message) . 
                           '&msg_type=' . urlencode($message_type));
                    exit;
                    
                } catch (Exception $e) {
                    $db->getConnection()->rollBack();
                    $message = 'خطأ: ' . $e->getMessage();
                    $message_type = 'error';
                }
            } else {
                // رسالة خطأ مفصلة
                if (empty($file_data)) {
                    $message = 'خطأ: الملف فارغ أو لا يحتوي على بيانات صحيحة';
                    if (empty($headers)) {
                        $message .= '<br>• لم يتم العثور على العناوين في الملف';
                    } elseif (count($headers) <= 1) {
                        $message .= '<br>• العناوين غير كافية (يجب أن يكون هناك عمود للرقم الجامعي وعدة أعمدة للمواد)<br>• العناوين الموجودة: ' . implode(', ', array_slice($headers, 0, 5));
                    } else {
                        $message .= '<br>• تم العثور على العناوين (' . count($headers) . ' عمود) لكن لا توجد صفوف بيانات<br>• العناوين: ' . implode(', ', array_slice($headers, 0, 10));
                    }
                } elseif (empty($headers)) {
                    $message = 'خطأ: لم يتم العثور على العناوين في الملف';
                } elseif (count($headers) <= 1) {
                    $message = 'خطأ: العناوين غير كافية. يجب أن يكون هناك عمود للرقم الجامعي وعدة أعمدة لأكواد المواد<br>• العناوين الموجودة: ' . implode(', ', $headers);
                } else {
                    $message = 'خطأ: الملف فارغ أو لا يحتوي على بيانات صحيحة';
                }
                $message_type = 'error';
            }
        }
    }
}

// جلب رسالة من URL
if (isset($_GET['msg']) && isset($_GET['msg_type'])) {
    $message = urldecode($_GET['msg']);
    $message_type = $_GET['msg_type'];
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>رفع درجات الفصل الدراسي من ملف Excel</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .upload-container {
            max-width: 800px;
            margin: 30px auto;
            padding: 30px;
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .form-steps {
            display: flex;
            gap: 10px;
            margin-bottom: 30px;
            flex-wrap: wrap;
        }
        
        .form-step {
            flex: 1;
            min-width: 150px;
        }
        
        .form-step label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #667eea;
        }
        
        .file-upload-area {
            border: 2px dashed #667eea;
            border-radius: 10px;
            padding: 30px;
            text-align: center;
            background: #f9f9f9;
            margin: 20px 0;
        }
        
        .file-upload-area:hover {
            background: #f0f0f0;
        }
        
        .instructions {
            background: #e3f2fd;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            border-right: 4px solid #2196F3;
        }
        
        .instructions h3 {
            color: #2196F3;
            margin-top: 0;
        }
        
        .instructions ol {
            margin: 10px 0;
            padding-right: 20px;
        }
        
        .instructions li {
            margin: 8px 0;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>رفع درجات الفصل الدراسي من ملف Excel/CSV</h2>
            <a href="dashboard.php" class="btn btn-secondary">العودة</a>
        </div>
        
        <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : ($message_type === 'error' ? 'error' : 'warning'); ?>" style="margin: 20px 0;">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>
        
        <div class="upload-container">
            <form method="GET" id="selection-form">
                <div class="form-steps">
                    <div class="form-step">
                        <label for="college_id">الكلية:</label>
                        <select name="college_id" id="college_id" class="form-group" style="width: 100%; padding: 10px;" onchange="this.form.submit()">
                            <option value="0">اختر الكلية</option>
                            <?php foreach ($available_colleges as $college): ?>
                                <option value="<?php echo $college['id']; ?>" <?php echo $college_id == $college['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($college['name_ar']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-step">
                        <label for="major_id">التخصص:</label>
                        <select name="major_id" id="major_id" class="form-group" style="width: 100%; padding: 10px;" onchange="this.form.submit()" <?php echo $college_id <= 0 ? 'disabled' : ''; ?>>
                            <option value="0">اختر التخصص</option>
                            <?php foreach ($available_majors as $major): ?>
                                <option value="<?php echo $major['id']; ?>" <?php echo $major_id == $major['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($major['name_ar']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-step">
                        <label for="semester_number">رقم الفصل:</label>
                        <select name="semester_number" id="semester_number" class="form-group" style="width: 100%; padding: 10px;" onchange="this.form.submit()" <?php echo $major_id <= 0 ? 'disabled' : ''; ?>>
                            <option value="0">اختر الفصل</option>
                            <?php foreach ($available_semester_numbers as $sem_num): ?>
                                <option value="<?php echo $sem_num; ?>" <?php echo $semester_number == $sem_num ? 'selected' : ''; ?>>
                                    الفصل <?php echo $sem_num; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-step">
                        <label for="academic_year">السنة الدراسية:</label>
                        <select name="academic_year" id="academic_year" class="form-group" style="width: 100%; padding: 10px;" onchange="this.form.submit()" <?php echo $semester_number <= 0 ? 'disabled' : ''; ?>>
                            <option value="">اختر السنة</option>
                            <?php foreach ($available_academic_years as $year): ?>
                                <option value="<?php echo htmlspecialchars($year); ?>" <?php echo $academic_year == $year ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($year); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
            </form>
            
            <?php if ($semester_id > 0): ?>
                <div class="instructions">
                    <h3>📋 تعليمات رفع الملف:</h3>
                    <ol>
                        <li>يجب أن يكون الملف بصيغة CSV أو Excel</li>
                        <li>العمود الأول يجب أن يحتوي على <strong>رقم الطالب</strong></li>
                        <li>باقي الأعمدة يجب أن تحتوي على <strong>أكواد المواد</strong> كعناوين</li>
                        <li>القيم في الخلايا هي <strong>الدرجات</strong> (يمكن أن تكون رقمية مثل 85 أو نصية مثل "بد", "غ")</li>
                        <li>مثال على شكل الملف:<br>
                            <table border="1" cellpadding="5" style="margin-top: 10px; border-collapse: collapse; font-size: 12px;">
                                <tr>
                                    <th>رقم الطالب</th>
                                    <th>CS101</th>
                                    <th>MATH101</th>
                                    <th>ENG101</th>
                                </tr>
                                <tr>
                                    <td>2023001</td>
                                    <td>85</td>
                                    <td>90</td>
                                    <td>75</td>
                                </tr>
                                <tr>
                                    <td>2023002</td>
                                    <td>80</td>
                                    <td>85</td>
                                    <td>بد</td>
                                </tr>
                            </table>
                        </li>
                    </ol>
                </div>
                
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="college_id" value="<?php echo $college_id; ?>">
                    <input type="hidden" name="major_id" value="<?php echo $major_id; ?>">
                    <input type="hidden" name="semester_number" value="<?php echo $semester_number; ?>">
                    <input type="hidden" name="academic_year" value="<?php echo htmlspecialchars($academic_year); ?>">
                    <input type="hidden" name="upload_grades" value="1">
                    
                    <div class="file-upload-area">
                        <label for="grades_file" style="cursor: pointer;">
                            <strong>📁 اختر ملف Excel أو CSV</strong><br>
                            <small>CSV, XLSX, XLS</small>
                        </label>
                        <input type="file" name="grades_file" id="grades_file" accept=".csv,.xlsx,.xls" required style="display: none;" onchange="document.getElementById('file-name').textContent = this.files[0] ? this.files[0].name : 'لم يتم اختيار ملف'">
                        <div id="file-name" style="margin-top: 10px; color: #666;"></div>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 20px;">📤 رفع الدرجات</button>
                </form>
            <?php else: ?>
                <div class="alert alert-warning" style="margin-top: 20px;">
                    يرجى اختيار الكلية والتخصص والفصل الدراسي والسنة الدراسية أولاً
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

