<?php
/**
 * سكريبت شامل لاختبار صفحة تسجيل الطلاب
 * Comprehensive Test Script for Student Registration Page
 */

// تعطيل عرض الأخطاء في HTML (سنعرضها بشكل منظم)
error_reporting(E_ALL);
ini_set('display_errors', '0');

?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>اختبار صفحة تسجيل الطلاب</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .header h1 {
            font-size: 28px;
            margin-bottom: 10px;
        }
        .content {
            padding: 30px;
        }
        .test-section {
            margin-bottom: 30px;
            padding: 20px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            background: #f9f9f9;
        }
        .test-section h2 {
            color: #667eea;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #667eea;
        }
        .test-item {
            margin: 15px 0;
            padding: 15px;
            background: white;
            border-radius: 5px;
            border-right: 4px solid #ddd;
        }
        .test-item.success {
            border-right-color: #4caf50;
            background: #f1f8f4;
        }
        .test-item.error {
            border-right-color: #f44336;
            background: #ffebee;
        }
        .test-item.warning {
            border-right-color: #ff9800;
            background: #fff3e0;
        }
        .test-item.info {
            border-right-color: #2196f3;
            background: #e3f2fd;
        }
        .status {
            display: inline-block;
            padding: 5px 15px;
            border-radius: 20px;
            font-weight: bold;
            margin-left: 10px;
            font-size: 14px;
        }
        .status.success {
            background: #4caf50;
            color: white;
        }
        .status.error {
            background: #f44336;
            color: white;
        }
        .status.warning {
            background: #ff9800;
            color: white;
        }
        .status.info {
            background: #2196f3;
            color: white;
        }
        .message {
            margin-top: 10px;
            padding: 10px;
            border-radius: 5px;
            font-size: 14px;
        }
        .message.error {
            background: #ffebee;
            color: #c62828;
            border-right: 3px solid #f44336;
        }
        .message.success {
            background: #f1f8f4;
            color: #2e7d32;
            border-right: 3px solid #4caf50;
        }
        .message.info {
            background: #e3f2fd;
            color: #1565c0;
            border-right: 3px solid #2196f3;
        }
        .summary {
            margin-top: 30px;
            padding: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 8px;
            text-align: center;
        }
        .summary h2 {
            margin-bottom: 15px;
        }
        .summary-stats {
            display: flex;
            justify-content: space-around;
            margin-top: 20px;
        }
        .stat {
            text-align: center;
        }
        .stat-number {
            font-size: 36px;
            font-weight: bold;
            margin-bottom: 5px;
        }
        .instructions {
            background: #fff3cd;
            border: 2px solid #ffc107;
            border-radius: 8px;
            padding: 20px;
            margin-top: 20px;
        }
        .instructions h3 {
            color: #856404;
            margin-bottom: 10px;
        }
        .instructions ul {
            margin-right: 20px;
            color: #856404;
        }
        .instructions li {
            margin: 8px 0;
        }
        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            color: #d63384;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔍 اختبار صفحة تسجيل الطلاب</h1>
            <p>نظام نتائج الجامعة - فحص شامل للاتصال والإعدادات</p>
        </div>
        
        <div class="content">
            <?php
            $results = [];
            $totalTests = 0;
            $passedTests = 0;
            $failedTests = 0;
            $warnings = 0;
            
            // دالة مساعدة لإضافة نتيجة اختبار
            function addTestResult(&$results, $title, $status, $message = '', $details = '') {
                global $totalTests, $passedTests, $failedTests, $warnings;
                $totalTests++;
                $results[] = [
                    'title' => $title,
                    'status' => $status, // success, error, warning, info
                    'message' => $message,
                    'details' => $details
                ];
                if ($status === 'success') $passedTests++;
                elseif ($status === 'error') $failedTests++;
                elseif ($status === 'warning') $warnings++;
            }
            
            // ========== 1. اختبار تحميل الملفات الأساسية ==========
            echo '<div class="test-section">';
            echo '<h2>1. اختبار تحميل الملفات الأساسية</h2>';
            
            // اختبار config.php
            if (file_exists(__DIR__ . '/config/config.php')) {
                try {
                    require_once __DIR__ . '/config/config.php';
                    addTestResult($results, 'تحميل config.php', 'success', 'تم تحميل ملف الإعدادات بنجاح');
                } catch (Exception $e) {
                    addTestResult($results, 'تحميل config.php', 'error', 'فشل تحميل ملف الإعدادات', $e->getMessage());
                }
            } else {
                addTestResult($results, 'تحميل config.php', 'error', 'ملف config.php غير موجود', 'المسار المتوقع: ' . __DIR__ . '/config/config.php');
            }
            
            // اختبار database.php
            if (file_exists(__DIR__ . '/config/database.php')) {
                addTestResult($results, 'وجود database.php', 'success', 'ملف إعدادات قاعدة البيانات موجود');
            } else {
                addTestResult($results, 'وجود database.php', 'error', 'ملف database.php غير موجود');
            }
            
            // اختبار students.php
            if (file_exists(__DIR__ . '/students.php')) {
                addTestResult($results, 'وجود students.php', 'success', 'صفحة تسجيل الطلاب موجودة');
            } else {
                addTestResult($results, 'وجود students.php', 'error', 'صفحة students.php غير موجودة');
            }
            
            echo '</div>';
            
            // ========== 2. اختبار إعدادات قاعدة البيانات ==========
            echo '<div class="test-section">';
            echo '<h2>2. اختبار إعدادات قاعدة البيانات</h2>';
            
            if (defined('DB_HOST') && defined('DB_USER') && defined('DB_NAME')) {
                addTestResult($results, 'تعريف الثوابت', 'success', 
                    'تم تعريف جميع الثوابت المطلوبة',
                    'DB_HOST: ' . DB_HOST . ', DB_USER: ' . DB_USER . ', DB_NAME: ' . DB_NAME);
                
                // التحقق من كلمة المرور
                if (defined('DB_PASS')) {
                    if (DB_PASS === '') {
                        addTestResult($results, 'كلمة مرور قاعدة البيانات', 'warning', 
                            'كلمة المرور فارغة - قد تحتاج إلى تعيين كلمة مرور في AppServ');
                    } else {
                        addTestResult($results, 'كلمة مرور قاعدة البيانات', 'info', 
                            'تم تعيين كلمة مرور (مخفية للأمان)');
                    }
                } else {
                    addTestResult($results, 'كلمة مرور قاعدة البيانات', 'error', 'DB_PASS غير معرّف');
                }
            } else {
                addTestResult($results, 'تعريف الثوابت', 'error', 'بعض الثوابت غير معرّفة');
            }
            
            echo '</div>';
            
            // ========== 3. اختبار الاتصال بقاعدة البيانات ==========
            echo '<div class="test-section">';
            echo '<h2>3. اختبار الاتصال بقاعدة البيانات</h2>';
            
            try {
                if (class_exists('Database')) {
                    $db = Database::getInstance();
                    $connection = $db->getConnection();
                    addTestResult($results, 'الاتصال بقاعدة البيانات', 'success', 
                        'تم الاتصال بقاعدة البيانات بنجاح!');
                    
                    // اختبار استعلام بسيط
                    try {
                        $db->query("SELECT 1");
                        addTestResult($results, 'تنفيذ استعلام تجريبي', 'success', 
                            'يمكن تنفيذ الاستعلامات بنجاح');
                    } catch (Exception $e) {
                        addTestResult($results, 'تنفيذ استعلام تجريبي', 'error', 
                            'فشل تنفيذ الاستعلام', $e->getMessage());
                    }
                } else {
                    addTestResult($results, 'الاتصال بقاعدة البيانات', 'error', 
                        'فئة Database غير موجودة');
                }
            } catch (PDOException $e) {
                $errorMsg = $e->getMessage();
                addTestResult($results, 'الاتصال بقاعدة البيانات', 'error', 
                    'فشل الاتصال بقاعدة البيانات', $errorMsg);
                
                // تحليل نوع الخطأ
                if (strpos($errorMsg, 'Access denied') !== false) {
                    addTestResult($results, 'تحليل الخطأ', 'warning', 
                        'خطأ في المصادقة - تحقق من اسم المستخدم وكلمة المرور',
                        'في AppServ، جرب: DB_PASS = "root" أو DB_PASS = ""');
                } elseif (strpos($errorMsg, 'Unknown database') !== false) {
                    addTestResult($results, 'تحليل الخطأ', 'warning', 
                        'قاعدة البيانات غير موجودة',
                        'قم بإنشاء قاعدة البيانات: ' . (defined('DB_NAME') ? DB_NAME : 'university_results'));
                } elseif (strpos($errorMsg, 'Connection refused') !== false) {
                    addTestResult($results, 'تحليل الخطأ', 'warning', 
                        'MySQL غير قيد التشغيل',
                        'تحقق من AppServ Control Panel وتأكد من تشغيل MySQL');
                }
            } catch (Exception $e) {
                addTestResult($results, 'الاتصال بقاعدة البيانات', 'error', 
                    'خطأ غير متوقع', $e->getMessage());
            }
            
            echo '</div>';
            
            // ========== 4. اختبار الجداول المطلوبة ==========
            echo '<div class="test-section">';
            echo '<h2>4. اختبار الجداول المطلوبة</h2>';
            
            $requiredTables = ['students', 'users', 'colleges', 'majors', 'courses'];
            
            try {
                if (isset($db) && $db) {
                    $existingTables = $db->fetchAll("SHOW TABLES");
                    $tableNames = [];
                    foreach ($existingTables as $table) {
                        $tableNames[] = array_values($table)[0];
                    }
                    
                    foreach ($requiredTables as $table) {
                        if (in_array($table, $tableNames)) {
                            // التحقق من وجود أعمدة أساسية
                            try {
                                $columns = $db->fetchAll("SHOW COLUMNS FROM `$table`");
                                addTestResult($results, "جدول $table", 'success', 
                                    "الجدول موجود ويحتوي على " . count($columns) . " عمود");
                            } catch (Exception $e) {
                                addTestResult($results, "جدول $table", 'error', 
                                    "خطأ في الوصول للجدول", $e->getMessage());
                            }
                        } else {
                            addTestResult($results, "جدول $table", 'error', 
                                "الجدول غير موجود في قاعدة البيانات");
                        }
                    }
                    
                    // التحقق من وجود بيانات
                    if (in_array('colleges', $tableNames)) {
                        try {
                            $collegeCount = $db->fetchOne("SELECT COUNT(*) as count FROM colleges");
                            if ($collegeCount['count'] > 0) {
                                addTestResult($results, 'بيانات الكليات', 'success', 
                                    "يوجد " . $collegeCount['count'] . " كلية في قاعدة البيانات");
                            } else {
                                addTestResult($results, 'بيانات الكليات', 'warning', 
                                    "لا توجد كليات في قاعدة البيانات - قد تحتاج إلى إضافة بيانات تجريبية");
                            }
                        } catch (Exception $e) {
                            addTestResult($results, 'بيانات الكليات', 'error', 
                                "خطأ في التحقق من بيانات الكليات", $e->getMessage());
                        }
                    }
                } else {
                    addTestResult($results, 'فحص الجداول', 'error', 
                        'لا يمكن فحص الجداول - فشل الاتصال بقاعدة البيانات');
                }
            } catch (Exception $e) {
                addTestResult($results, 'فحص الجداول', 'error', 
                    'خطأ في فحص الجداول', $e->getMessage());
            }
            
            echo '</div>';
            
            // ========== 5. اختبار الصلاحيات والملفات ==========
            echo '<div class="test-section">';
            echo '<h2>5. اختبار الصلاحيات والملفات</h2>';
            
            // اختبار مجلدات الكتابة
            $writableDirs = [
                'uploads' => UPLOAD_DIR ?? __DIR__ . '/uploads/',
                'certificates' => CERTIFICATES_DIR ?? __DIR__ . '/certificates/',
                'logs' => LOG_DIR ?? __DIR__ . '/logs/'
            ];
            
            foreach ($writableDirs as $name => $path) {
                if (file_exists($path)) {
                    if (is_writable($path)) {
                        addTestResult($results, "مجلد $name", 'success', 
                            "المجلد موجود وقابل للكتابة");
                    } else {
                        addTestResult($results, "مجلد $name", 'warning', 
                            "المجلد موجود لكن غير قابل للكتابة",
                            "قم بتغيير الصلاحيات: chmod 777 $path");
                    }
                } else {
                    addTestResult($results, "مجلد $name", 'warning', 
                        "المجلد غير موجود - سيتم إنشاؤه تلقائياً");
                }
            }
            
            echo '</div>';
            
            // ========== عرض النتائج ==========
            foreach ($results as $result) {
                echo '<div class="test-item ' . $result['status'] . '">';
                echo '<strong>' . htmlspecialchars($result['title']) . '</strong>';
                echo '<span class="status ' . $result['status'] . '">';
                if ($result['status'] === 'success') echo '✓ نجح';
                elseif ($result['status'] === 'error') echo '✗ فشل';
                elseif ($result['status'] === 'warning') echo '⚠ تحذير';
                else echo 'ℹ معلومات';
                echo '</span>';
                
                if ($result['message']) {
                    echo '<div class="message ' . $result['status'] . '">';
                    echo htmlspecialchars($result['message']);
                    echo '</div>';
                }
                
                if ($result['details']) {
                    echo '<div style="margin-top: 8px; font-size: 12px; color: #666; font-family: monospace;">';
                    echo htmlspecialchars($result['details']);
                    echo '</div>';
                }
                
                echo '</div>';
            }
            
            // ========== الملخص ==========
            echo '<div class="summary">';
            echo '<h2>ملخص الاختبارات</h2>';
            echo '<div class="summary-stats">';
            echo '<div class="stat">';
            echo '<div class="stat-number">' . $totalTests . '</div>';
            echo '<div>إجمالي الاختبارات</div>';
            echo '</div>';
            echo '<div class="stat">';
            echo '<div class="stat-number" style="color: #4caf50;">' . $passedTests . '</div>';
            echo '<div>نجحت</div>';
            echo '</div>';
            echo '<div class="stat">';
            echo '<div class="stat-number" style="color: #f44336;">' . $failedTests . '</div>';
            echo '<div>فشلت</div>';
            echo '</div>';
            echo '<div class="stat">';
            echo '<div class="stat-number" style="color: #ff9800;">' . $warnings . '</div>';
            echo '<div>تحذيرات</div>';
            echo '</div>';
            echo '</div>';
            echo '</div>';
            
            // ========== إرشادات حل المشاكل ==========
            if ($failedTests > 0 || $warnings > 0) {
                echo '<div class="instructions">';
                echo '<h3>🔧 إرشادات حل المشاكل</h3>';
                echo '<ul>';
                
                if ($failedTests > 0) {
                    echo '<li><strong>مشاكل الاتصال بقاعدة البيانات:</strong><br>';
                    echo '1. تأكد من تشغيل MySQL من AppServ Control Panel<br>';
                    echo '2. افتح ملف <code>config/database.php</code> وتحقق من:<br>';
                    echo '   - <code>DB_HOST</code> = "localhost"<br>';
                    echo '   - <code>DB_USER</code> = "root"<br>';
                    echo '   - <code>DB_PASS</code> = "root" أو "" (حسب إعدادات AppServ)<br>';
                    echo '   - <code>DB_NAME</code> = "university_results"<br>';
                    echo '3. تأكد من إنشاء قاعدة البيانات: <code>mysql -u root -p < database/schema.sql</code><br>';
                    echo '4. أو استورد ملف <code>database/schema.sql</code> من phpMyAdmin</li>';
                }
                
                if ($warnings > 0) {
                    echo '<li><strong>مشاكل الصلاحيات:</strong><br>';
                    echo 'تأكد من أن المجلدات قابلة للكتابة. في Windows، عادة لا توجد مشاكل.<br>';
                    echo 'في Linux/Mac: <code>chmod 777 uploads certificates logs</code></li>';
                }
                
                echo '<li><strong>اختبار الاتصال يدوياً:</strong><br>';
                echo 'يمكنك فتح phpMyAdmin والتحقق من:<br>';
                echo '1. وجود قاعدة البيانات "university_results"<br>';
                echo '2. وجود الجداول المطلوبة<br>';
                echo '3. إمكانية تسجيل الدخول بـ root</li>';
                
                echo '</ul>';
                echo '</div>';
            } else {
                echo '<div class="instructions" style="background: #d4edda; border-color: #28a745;">';
                echo '<h3 style="color: #155724;">✅ كل شيء يعمل بشكل صحيح!</h3>';
                echo '<p style="color: #155724; margin-top: 10px;">يمكنك الآن استخدام صفحة تسجيل الطلاب بدون مشاكل.</p>';
                echo '</div>';
            }
            ?>
        </div>
    </div>
</body>
</html>

