<?php
// تفعيل عرض الأخطاء مؤقتاً للتشخيص
error_reporting(E_ALL);
ini_set('display_errors', '1');

try {
    require_once __DIR__ . '/config/config.php';
    require_once __DIR__ . '/classes/Auth.php';
} catch (Exception $e) {
    die('خطأ في تحميل الملفات: ' . htmlspecialchars($e->getMessage()));
}

try {
    $auth = new Auth();
} catch (Exception $e) {
    die('خطأ في تهيئة Auth: ' . htmlspecialchars($e->getMessage()));
}

if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin') && !$auth->hasRole('registrar'))) {
    header('Location: index.php');
    exit;
}

try {
    $db = Database::getInstance();
} catch (Exception $e) {
    die('خطأ في الاتصال بقاعدة البيانات: ' . htmlspecialchars($e->getMessage()));
}

// معالجة AJAX requests
if (isset($_GET['action']) && $_GET['action'] == 'get_majors' && isset($_GET['college_id'])) {
    header('Content-Type: application/json');
    $college_id = (int)$_GET['college_id'];
    $majors = $db->fetchAll(
        "SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar",
        [$college_id]
    );
    echo json_encode($majors);
    exit;
}

$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$message = '';

// معالجة رفع ملف Excel/CSV
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'upload_excel') {
    if (isset($_FILES['excel_file']) && $_FILES['excel_file']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['excel_file'];
        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        // التحقق من حجم الملف (10MB)
        $max_size = 10 * 1024 * 1024; // 10MB
        if ($file['size'] > $max_size) {
            $message = '<div class="alert alert-error">خطأ: حجم الملف كبير جداً. الحد الأقصى 10MB</div>';
        }
        // التحقق من نوع الملف
        elseif (!in_array($file_ext, ['csv', 'xlsx', 'xls'])) {
            $message = '<div class="alert alert-error">خطأ: نوع الملف غير مدعوم. يرجى رفع ملف CSV أو Excel (.xlsx, .xls)</div>';
        } else {
            try {
                require_once __DIR__ . '/classes/ExcelReader.php';
                $excelReader = new ExcelReader();
                
                $students_data = $excelReader->readFile($file['tmp_name'], $file_ext);
                
                if ($students_data === false || empty($students_data)) {
                    $message = '<div class="alert alert-error">خطأ: فشل قراءة الملف أو الملف فارغ. يرجى التحقق من صحة الملف والتنسيق</div>';
                } else {
                    $success_count = 0;
                    $error_count = 0;
                    $duplicate_count = 0;
                    $errors = [];
                    
                    // الحصول على الكلية الافتراضية
                    $default_college_id = $auth->hasRole('admin') ? (int)($_POST['college_id'] ?? 0) : ($_SESSION['college_id'] ?? 0);
                    
                    if ($default_college_id <= 0) {
                        $message = '<div class="alert alert-error">خطأ: يجب تحديد كلية</div>';
                    } else {
                        // بدء معاملة قاعدة البيانات
                        try {
                            $db->getConnection()->beginTransaction();
                            
                            foreach ($students_data as $row_index => $row) {
                                // قراءة الحقول - البحث في عدة مفاتيح محتملة
                                $student_number = '';
                                if (isset($row['student_number']) && !empty(trim($row['student_number']))) {
                                    $student_number = trim($row['student_number']);
                                } elseif (isset($row['رقم_جامعي']) && !empty(trim($row['رقم_جامعي']))) {
                                    $student_number = trim($row['رقم_جامعي']);
                                } elseif (isset($row['الرقم الجامعي']) && !empty(trim($row['الرقم الجامعي']))) {
                                    $student_number = trim($row['الرقم الجامعي']);
                                }
                                
                                $full_name_ar = '';
                                if (isset($row['full_name_ar']) && !empty(trim($row['full_name_ar']))) {
                                    $full_name_ar = trim($row['full_name_ar']);
                                } elseif (isset($row['name_ar']) && !empty(trim($row['name_ar']))) {
                                    $full_name_ar = trim($row['name_ar']);
                                } elseif (isset($row['اسم_عربي']) && !empty(trim($row['اسم_عربي']))) {
                                    $full_name_ar = trim($row['اسم_عربي']);
                                } elseif (isset($row['الاسم العربي']) && !empty(trim($row['الاسم العربي']))) {
                                    $full_name_ar = trim($row['الاسم العربي']);
                                } elseif (isset($row['اسم']) && !empty(trim($row['اسم']))) {
                                    $full_name_ar = trim($row['اسم']);
                                } elseif (isset($row['الاسم']) && !empty(trim($row['الاسم']))) {
                                    $full_name_ar = trim($row['الاسم']);
                                }
                                
                                $national_id = trim($row['national_id'] ?? $row['رقم_وطني'] ?? $row['الرقم الوطني'] ?? '');
                                $full_name_en = trim($row['full_name_en'] ?? $row['name_en'] ?? $row['اسم_إنجليزي'] ?? $row['الاسم الإنجليزي'] ?? '');
                                $email = trim($row['email'] ?? $row['بريد'] ?? $row['البريد الإلكتروني'] ?? '');
                                $phone = trim($row['phone'] ?? $row['هاتف'] ?? $row['الهاتف'] ?? '');
                                $major_code = trim($row['major_code'] ?? $row['رمز_تخصص'] ?? $row['رمز التخصص'] ?? '');
                                $major_id_from_file = isset($row['major_id']) ? (int)$row['major_id'] : 0;
                                $enrollment_date = trim($row['enrollment_date'] ?? $row['تاريخ_تسجيل'] ?? $row['تاريخ التسجيل'] ?? date('Y-m-d'));
                                
                                // التحقق من الحقول المطلوبة
                                if (empty($student_number) || empty($full_name_ar)) {
                                    $error_count++;
                                    $row_number = $row_index + 2; // +2 لأن الفهرس يبدأ من 0 والسطر الأول هو العناوين
                                    
                                    // إنشاء رسالة خطأ تفصيلية للمساعدة في التصحيح
                                    $error_msg = "السطر $row_number: ";
                                    
                                    if (empty($student_number) && empty($full_name_ar)) {
                                        $error_msg .= "لا يوجد رقم جامعي ولا اسم";
                                    } elseif (empty($student_number)) {
                                        $error_msg .= "لا يوجد رقم جامعي (الاسم: " . htmlspecialchars($full_name_ar) . ")";
                                    } else {
                                        $error_msg .= "لا يوجد اسم (الرقم الجامعي: " . htmlspecialchars($student_number) . ")";
                                    }
                                    
                                    // في وضع التطوير، أضف معلومات إضافية
                                    if (count($errors) < 3) {
                                        $available_keys = array_keys($row);
                                        $error_msg .= " | الحقول الموجودة في السطر: " . implode(', ', array_slice($available_keys, 0, 10));
                                        if (count($available_keys) > 10) {
                                            $error_msg .= "... (و" . (count($available_keys) - 10) . " حقول أخرى)";
                                        }
                                    }
                                    
                                    $errors[] = $error_msg;
                                    continue;
                                }
                                
                                // البحث عن التخصص
                                $selected_major_id = 0;
                                if ($major_id_from_file > 0) {
                                    $selected_major_id = $major_id_from_file;
                                } elseif (!empty($major_code)) {
                                    $major = $db->fetchOne(
                                        "SELECT * FROM majors WHERE code = ? AND college_id = ?",
                                        [$major_code, $default_college_id]
                                    );
                                    if ($major) {
                                        $selected_major_id = $major['id'];
                                    }
                                }
                                
                                // إذا لم يتم العثور على التخصص، استخدام التخصص المحدد من النموذج
                                if ($selected_major_id <= 0) {
                                    $selected_major_id = (int)($_POST['major_id'] ?? 0);
                                }
                                
                                if ($selected_major_id <= 0) {
                                    $error_count++;
                                    $errors[] = "الطالب $student_number: لم يتم العثور على التخصص";
                                    continue;
                                }
                                
                                // التحقق من الصلاحيات
                                if (!$auth->belongsToCollege($default_college_id) && !$auth->hasRole('admin')) {
                                    $error_count++;
                                    $errors[] = "الطالب $student_number: ليس لديك صلاحية لإضافة طالب في هذه الكلية";
                                    continue;
                                }
                                
                                // التحقق من عدم وجود طالب بنفس الرقم الجامعي
                                $existing = $db->fetchOne(
                                    "SELECT * FROM students WHERE student_number = ?",
                                    [$student_number]
                                );
                                
                                if ($existing) {
                                    $duplicate_count++;
                                    continue;
                                }
                                
                                // التحقق من الرقم الوطني إذا كان موجوداً
                                if (!empty($national_id)) {
                                    $existing_nid = $db->fetchOne(
                                        "SELECT * FROM students WHERE national_id = ? AND national_id != ''",
                                        [$national_id]
                                    );
                                    if ($existing_nid) {
                                        $duplicate_count++;
                                        continue;
                                    }
                                }
                                
                                // إنشاء مستخدم للطالب
                                $username = 'STU' . $student_number;
                                $password = password_hash($student_number, PASSWORD_DEFAULT);
                                
                                try {
                                    // إضافة المستخدم
                                    $db->query(
                                        "INSERT INTO users (college_id, username, password, full_name_ar, full_name_en, email, role) 
                                         VALUES (?, ?, ?, ?, ?, ?, 'student')",
                                        [$default_college_id, $username, $password, $full_name_ar, $full_name_en, $email]
                                    );
                                    
                                    $user_id = $db->getConnection()->lastInsertId();
                                    
                                    // إضافة الطالب
                                    $db->query(
                                        "INSERT INTO students (college_id, major_id, student_number, national_id, full_name_ar, full_name_en, email, phone, enrollment_date, user_id, status) 
                                         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'active')",
                                        [$default_college_id, $selected_major_id, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $enrollment_date, $user_id]
                                    );
                                    
                                    $success_count++;
                                } catch (PDOException $e) {
                                    $error_count++;
                                    $error_msg = $e->getMessage();
                                    if (strpos($error_msg, 'Duplicate entry') !== false || strpos($error_msg, '1062') !== false) {
                                        $duplicate_count++;
                                    } else {
                                        $errors[] = "الطالب $student_number: " . $error_msg;
                                        error_log("Error adding student from Excel: " . $error_msg);
                                    }
                                }
                            }
                            
                            // تأكيد المعاملة
                            $db->getConnection()->commit();
                            
                        } catch (Exception $e) {
                            // إلغاء المعاملة في حالة الخطأ
                            $db->getConnection()->rollBack();
                            $message = '<div class="alert alert-error">خطأ في المعاملة: ' . htmlspecialchars($e->getMessage()) . '</div>';
                            error_log("Transaction error: " . $e->getMessage());
                        }
                        
                        // بناء رسالة النتيجة
                        if ($success_count > 0) {
                            $message = '<div class="alert alert-success">✓ تم إضافة ' . $success_count . ' طالب بنجاح من الملف</div>';
                        }
                        if ($duplicate_count > 0) {
                            $message .= '<div class="alert alert-info">⚠ ' . $duplicate_count . ' طالب موجودون مسبقاً (تم تخطيهم)</div>';
                        }
                        if ($error_count > 0) {
                            $message .= '<div class="alert alert-warning">⚠ ' . $error_count . ' خطأ في الإضافة</div>';
                            if (count($errors) > 0 && count($errors) <= 10) {
                                $message .= '<div class="alert alert-info" style="margin-top:10px;"><small>' . implode('<br>', array_map('htmlspecialchars', $errors)) . '</small></div>';
                            }
                        }
                    }
                }
            } catch (Exception $e) {
                $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '<br>يرجى استخدام ملف CSV أو تثبيت مكتبة PhpSpreadsheet لقراءة ملفات Excel</div>';
            }
        }
    } elseif (isset($_FILES['excel_file'])) {
        $error_msg = 'خطأ غير معروف';
        switch ($_FILES['excel_file']['error']) {
            case UPLOAD_ERR_INI_SIZE:
            case UPLOAD_ERR_FORM_SIZE:
                $error_msg = 'حجم الملف كبير جداً';
                break;
            case UPLOAD_ERR_PARTIAL:
                $error_msg = 'تم رفع جزء من الملف فقط';
                break;
            case UPLOAD_ERR_NO_FILE:
                $error_msg = 'لم يتم اختيار ملف';
                break;
            case UPLOAD_ERR_NO_TMP_DIR:
                $error_msg = 'مجلد الملفات المؤقتة غير موجود';
                break;
            case UPLOAD_ERR_CANT_WRITE:
                $error_msg = 'فشل كتابة الملف على القرص';
                break;
        }
        $message = '<div class="alert alert-error">' . $error_msg . '</div>';
    } else {
        $message = '<div class="alert alert-error">خطأ: لم يتم رفع الملف</div>';
    }
}

// معالجة الإضافة
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add') {
    $student_number = $_POST['student_number'] ?? '';
    $national_id = $_POST['national_id'] ?? '';
    $full_name_ar = $_POST['full_name_ar'] ?? '';
    $full_name_en = $_POST['full_name_en'] ?? '';
    $email = $_POST['email'] ?? '';
    $phone = $_POST['phone'] ?? '';
    $college_id = (int)($_POST['college_id'] ?? $_SESSION['college_id']);
    $major_id_post = (int)($_POST['major_id'] ?? 0);
    $enrollment_date = $_POST['enrollment_date'] ?? date('Y-m-d');
    
    if ($student_number && $full_name_ar && $major_id_post) {
        // التحقق من الصلاحية
        if (!$auth->belongsToCollege($college_id) && !$auth->hasRole('admin')) {
            $message = '<div class="alert alert-error">ليس لديك صلاحية لإضافة طالب في هذه الكلية</div>';
        } else {
            try {
                // إنشاء مستخدم للطالب
                $username = 'STU' . $student_number;
                $password = password_hash($student_number, PASSWORD_DEFAULT);
                
                $db->query(
                    "INSERT INTO users (college_id, username, password, full_name_ar, full_name_en, email, role) 
                     VALUES (?, ?, ?, ?, ?, ?, 'student')",
                    [$college_id, $username, $password, $full_name_ar, $full_name_en, $email]
                );
                
                $user_id = $db->getConnection()->lastInsertId();
                
                // إضافة الطالب
                $db->query(
                    "INSERT INTO students (college_id, major_id, student_number, national_id, full_name_ar, full_name_en, email, phone, enrollment_date, user_id) 
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                    [$college_id, $major_id_post, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $enrollment_date, $user_id]
                );
                
                // تسجيل نشاط إضافة الطالب
                require_once __DIR__ . '/classes/ActivityLogger.php';
                $logger = new ActivityLogger();
                $student_id = $db->getConnection()->lastInsertId();
                $logger->log('create', 'student', $student_id, 'إضافة طالب جديد: ' . $full_name_ar . ' (الرقم الجامعي: ' . $student_number . ')');
                
                $message = '<div class="alert alert-success">تم إضافة الطالب بنجاح</div>';
            } catch (PDOException $e) {
                $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
        }
    }
}

// معالجة الحذف
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $student_id = (int)($_POST['student_id'] ?? 0);
    
    if ($student_id > 0) {
        try {
            // جلب معلومات الطالب للتحقق من الصلاحيات
            $student = $db->fetchOne(
                "SELECT s.* 
                 FROM students s 
                 WHERE s.id = ?",
                [$student_id]
            );
            
            if (!$student) {
                $message = '<div class="alert alert-error">الطالب غير موجود</div>';
            } else {
                // التحقق من الصلاحية
                if (!$auth->belongsToCollege($student['college_id']) && !$auth->hasRole('admin')) {
                    $message = '<div class="alert alert-error">ليس لديك صلاحية لحذف طالب من هذه الكلية</div>';
                } else {
                    // بدء معاملة قاعدة البيانات
                    $db->getConnection()->beginTransaction();
                    
                    try {
                        // حذف المستخدم المرتبط إذا كان موجوداً
                        if ($student['user_id']) {
                            $db->query("DELETE FROM users WHERE id = ?", [$student['user_id']]);
                        }
                        
                        // حذف الطالب (سيتم حذف البيانات المرتبطة تلقائياً بسبب CASCADE)
                        $db->query("DELETE FROM students WHERE id = ?", [$student_id]);
                        
                        // تسجيل نشاط حذف الطالب
                        require_once __DIR__ . '/classes/ActivityLogger.php';
                        $logger = new ActivityLogger();
                        $logger->log('delete', 'student', $student_id, 'حذف طالب: ' . $student['full_name_ar'] . ' (الرقم الجامعي: ' . $student['student_number'] . ')');
                        
                        // تأكيد المعاملة
                        $db->getConnection()->commit();
                        
                        $message = '<div class="alert alert-success">تم حذف الطالب بنجاح</div>';
                    } catch (Exception $e) {
                        // إلغاء المعاملة في حالة الخطأ
                        $db->getConnection()->rollBack();
                        throw $e;
                    }
                }
            }
        } catch (PDOException $e) {
            $message = '<div class="alert alert-error">خطأ في حذف الطالب: ' . htmlspecialchars($e->getMessage()) . '</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    } else {
        $message = '<div class="alert alert-error">معرف الطالب غير صحيح</div>';
    }
}

// معالجة حذف جميع الطلاب (فقط للـ admin)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete_all') {
    // التحقق من الصلاحيات - فقط admin يمكنه حذف جميع الطلاب
    if (!$auth->hasRole('admin')) {
        $message = '<div class="alert alert-error">⚠️ خطأ: ليس لديك صلاحية لحذف جميع الطلاب. هذه العملية متاحة فقط للمدير العام.</div>';
    } else {
        // التحقق من تأكيد الحذف
        $confirm_text = trim($_POST['confirm_text'] ?? '');
        $expected_text = 'حذف جميع الطلاب';
        
        if ($confirm_text !== $expected_text) {
            $message = '<div class="alert alert-error">⚠️ خطأ: لم يتم تأكيد العملية. يجب كتابة "حذف جميع الطلاب" للتأكيد.</div>';
        } else {
            try {
                // جلب عدد الطلاب قبل الحذف
                $total_students = $db->fetchOne("SELECT COUNT(*) as count FROM students")['count'];
                
                if ($total_students == 0) {
                    $message = '<div class="alert alert-info">لا يوجد طلاب لحذفهم</div>';
                } else {
                    // بدء معاملة قاعدة البيانات
                    $db->getConnection()->beginTransaction();
                    
                    try {
                        // جلب جميع معرفات المستخدمين المرتبطين بالطلاب
                        $students_with_users = $db->fetchAll(
                            "SELECT DISTINCT user_id FROM students WHERE user_id IS NOT NULL"
                        );
                        
                        // حذف جميع المستخدمين المرتبطين بالطلاب
                        if (!empty($students_with_users)) {
                            $user_ids = array_column($students_with_users, 'user_id');
                            $placeholders = implode(',', array_fill(0, count($user_ids), '?'));
                            $db->query("DELETE FROM users WHERE id IN ($placeholders)", $user_ids);
                        }
                        
                        // حذف جميع الطلاب (سيتم حذف البيانات المرتبطة تلقائياً بسبب CASCADE)
                        // البيانات المرتبطة تشمل: student_courses, grades, semester_gpas, cumulative_gpas, etc.
                        $db->query("DELETE FROM students");
                        
                        // تسجيل نشاط حذف جميع الطلاب
                        require_once __DIR__ . '/classes/ActivityLogger.php';
                        $logger = new ActivityLogger();
                        $logger->log('delete', 'student', null, 'حذف جميع الطلاب (' . $total_students . ' طالب) - تم بواسطة: ' . ($_SESSION['username'] ?? 'غير معروف'));
                        
                        // تأكيد المعاملة
                        $db->getConnection()->commit();
                        
                        $message = '<div class="alert alert-success">✓ تم حذف جميع الطلاب بنجاح (' . $total_students . ' طالب)</div>';
                        $message .= '<div class="alert alert-info">⚠️ تم حذف جميع البيانات المرتبطة (التسجيلات، الدرجات، المعدلات، إلخ)</div>';
                    } catch (Exception $e) {
                        // إلغاء المعاملة في حالة الخطأ
                        $db->getConnection()->rollBack();
                        throw $e;
                    }
                }
            } catch (PDOException $e) {
                $message = '<div class="alert alert-error">خطأ في حذف جميع الطلاب: ' . htmlspecialchars($e->getMessage()) . '</div>';
            } catch (Exception $e) {
                $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
        }
    }
}

// جلب الطلاب مع معالجة الأخطاء
try {
    if ($auth->hasRole('admin')) {
        if ($major_id) {
            $students = $db->fetchAll(
                "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name,
                        sem.semester_number, sem.academic_year
                 FROM students s 
                 JOIN majors m ON s.major_id = m.id 
                 JOIN colleges c ON s.college_id = c.id 
                 LEFT JOIN (
                     SELECT sc.student_id, s.semester_number, s.academic_year,
                            ROW_NUMBER() OVER (PARTITION BY sc.student_id ORDER BY s.academic_year DESC, s.semester_number DESC) as rn
                     FROM student_courses sc
                     JOIN semesters s ON sc.semester_id = s.id
                 ) sem ON sem.student_id = s.id AND sem.rn = 1
                 WHERE s.major_id = ?
                 ORDER BY s.student_number",
                [$major_id]
            );
        } else {
            $students = $db->fetchAll(
                "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name,
                        sem.semester_number, sem.academic_year
                 FROM students s 
                 JOIN majors m ON s.major_id = m.id 
                 JOIN colleges c ON s.college_id = c.id 
                 LEFT JOIN (
                     SELECT sc.student_id, s.semester_number, s.academic_year,
                            ROW_NUMBER() OVER (PARTITION BY sc.student_id ORDER BY s.academic_year DESC, s.semester_number DESC) as rn
                     FROM student_courses sc
                     JOIN semesters s ON sc.semester_id = s.id
                 ) sem ON sem.student_id = s.id AND sem.rn = 1
                 ORDER BY s.student_number"
            );
        }
    } else {
        $college_id = $_SESSION['college_id'] ?? 0;
        if ($major_id) {
            $students = $db->fetchAll(
                "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name,
                        sem.semester_number, sem.academic_year
                 FROM students s 
                 JOIN majors m ON s.major_id = m.id 
                 JOIN colleges c ON s.college_id = c.id 
                 LEFT JOIN (
                     SELECT sc.student_id, s.semester_number, s.academic_year,
                            ROW_NUMBER() OVER (PARTITION BY sc.student_id ORDER BY s.academic_year DESC, s.semester_number DESC) as rn
                     FROM student_courses sc
                     JOIN semesters s ON sc.semester_id = s.id
                 ) sem ON sem.student_id = s.id AND sem.rn = 1
                 WHERE s.college_id = ? AND s.major_id = ?
                 ORDER BY s.student_number",
                [$college_id, $major_id]
            );
        } else {
            $students = $db->fetchAll(
                "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name,
                        sem.semester_number, sem.academic_year
                 FROM students s 
                 JOIN majors m ON s.major_id = m.id 
                 JOIN colleges c ON s.college_id = c.id 
                 LEFT JOIN (
                     SELECT sc.student_id, s.semester_number, s.academic_year,
                            ROW_NUMBER() OVER (PARTITION BY sc.student_id ORDER BY s.academic_year DESC, s.semester_number DESC) as rn
                     FROM student_courses sc
                     JOIN semesters s ON sc.semester_id = s.id
                 ) sem ON sem.student_id = s.id AND sem.rn = 1
                 WHERE s.college_id = ?
                 ORDER BY s.student_number",
                [$college_id]
            );
        }
    }
} catch (Exception $e) {
    // في حالة عدم دعم ROW_NUMBER، استخدام استعلام بديل
    try {
        if ($auth->hasRole('admin')) {
            if ($major_id) {
                $students = $db->fetchAll(
                    "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name,
                            (SELECT sem.semester_number FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as semester_number,
                            (SELECT sem.academic_year FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as academic_year
                     FROM students s 
                     JOIN majors m ON s.major_id = m.id 
                     JOIN colleges c ON s.college_id = c.id 
                     WHERE s.major_id = ?
                     ORDER BY s.student_number",
                    [$major_id]
                );
            } else {
                $students = $db->fetchAll(
                    "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name,
                            (SELECT sem.semester_number FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as semester_number,
                            (SELECT sem.academic_year FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as academic_year
                     FROM students s 
                     JOIN majors m ON s.major_id = m.id 
                     JOIN colleges c ON s.college_id = c.id 
                     ORDER BY s.student_number"
                );
            }
        } else {
            $college_id = $_SESSION['college_id'] ?? 0;
            if ($major_id) {
                $students = $db->fetchAll(
                    "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name,
                            (SELECT sem.semester_number FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as semester_number,
                            (SELECT sem.academic_year FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as academic_year
                     FROM students s 
                     JOIN majors m ON s.major_id = m.id 
                     JOIN colleges c ON s.college_id = c.id 
                     WHERE s.college_id = ? AND s.major_id = ?
                     ORDER BY s.student_number",
                    [$college_id, $major_id]
                );
            } else {
                $students = $db->fetchAll(
                    "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name,
                            (SELECT sem.semester_number FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as semester_number,
                            (SELECT sem.academic_year FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as academic_year
                     FROM students s 
                     JOIN majors m ON s.major_id = m.id 
                     JOIN colleges c ON s.college_id = c.id 
                     WHERE s.college_id = ?
                     ORDER BY s.student_number",
                    [$college_id]
                );
            }
        }
    } catch (Exception $e2) {
        $students = [];
        $message = '<div class="alert alert-error">خطأ في جلب بيانات الطلاب: ' . htmlspecialchars($e2->getMessage()) . '</div>';
    }
}

// جلب التخصصات - للـ admin نبدأ بقائمة فارغة، سيتم تحميلها حسب الكلية المختارة
try {
    if ($auth->hasRole('admin')) {
        // للـ admin، نبدأ بقائمة فارغة - سيتم تحميل التخصصات ديناميكياً حسب الكلية المختارة
        $majors = [];
    } else {
        $college_id = $_SESSION['college_id'] ?? 0;
        $majors = $db->fetchAll(
            "SELECT m.* FROM majors m 
             JOIN colleges c ON m.college_id = c.id 
             WHERE c.id = ?
             ORDER BY m.name_ar",
            [$college_id]
        );
    }
} catch (Exception $e) {
    $majors = [];
    $message .= '<div class="alert alert-error">خطأ في جلب التخصصات: ' . htmlspecialchars($e->getMessage()) . '</div>';
}

// جلب الكليات
try {
    if ($auth->hasRole('admin')) {
        $colleges = $db->fetchAll("SELECT * FROM colleges ORDER BY name_ar");
    } else {
        $college_id = $_SESSION['college_id'] ?? 0;
        $college_name = $_SESSION['college_name'] ?? 'غير محدد';
        $colleges = [['id' => $college_id, 'name_ar' => $college_name]];
    }
} catch (Exception $e) {
    $colleges = [];
    $message .= '<div class="alert alert-error">خطأ في جلب الكليات: ' . htmlspecialchars($e->getMessage()) . '</div>';
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة الطلاب</title>
    <?php include __DIR__ . '/includes/compatibility.php'; ?>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>إدارة الطلاب</h2>
            <div>
                <button onclick="document.getElementById('addForm').style.display='block'" class="btn btn-primary">إضافة طالب</button>
                <button onclick="document.getElementById('uploadForm').style.display='block'" class="btn btn-success">رفع طلاب من Excel</button>
                <?php if ($auth->hasRole('admin')): ?>
                    <button onclick="document.getElementById('deleteAllForm').style.display='block'" class="btn btn-danger">🗑️ حذف جميع الطلاب</button>
                <?php endif; ?>
            </div>
        </div>
        
        <?php echo $message; ?>
        
        <!-- قسم رفع ملف Excel/CSV -->
        <div id="uploadForm" class="card" style="display:none; margin-bottom: 20px; border: 2px solid #4CAF50;">
            <div class="card-header" style="background-color: #4CAF50; color: white;">
                <h3>📥 رفع طلاب من ملف Excel/CSV</h3>
            </div>
            <div style="padding: 20px;">
                <form method="POST" enctype="multipart/form-data" style="margin-bottom: 15px;">
                    <input type="hidden" name="action" value="upload_excel">
                    
                    <?php if ($auth->hasRole('admin')): ?>
                    <div class="form-group">
                        <label>الكلية:</label>
                        <select name="college_id" id="upload_college_id" required onchange="loadMajorsForUpload()">
                            <option value="">-- اختر الكلية --</option>
                            <?php foreach ($colleges as $college): ?>
                                <option value="<?php echo $college['id']; ?>"><?php echo htmlspecialchars($college['name_ar']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php else: ?>
                        <input type="hidden" name="college_id" id="upload_college_id" value="<?php echo $_SESSION['college_id']; ?>">
                    <?php endif; ?>
                    
                    <div class="form-group">
                        <label>التخصص (اختياري - سيتم استخدامه إذا لم يكن موجوداً في الملف):</label>
                        <select name="major_id" id="upload_major_id" disabled>
                            <option value="">-- اختر الكلية أولاً --</option>
                            <?php foreach ($majors as $major): ?>
                                <option value="<?php echo $major['id']; ?>"><?php echo htmlspecialchars($major['name_ar']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="excel_file" style="display: block; margin-bottom: 8px; font-weight: bold;">
                            اختر ملف Excel أو CSV:
                        </label>
                        <input type="file" 
                               name="excel_file" 
                               id="excel_file" 
                               accept=".csv,.xlsx,.xls"
                               required
                               style="padding: 8px; width: 100%; max-width: 400px; border: 1px solid #ddd; border-radius: 4px;">
                        <small style="display: block; margin-top: 5px; color: #666;">
                            الصيغ المدعومة: CSV, Excel (.xlsx, .xls) - الحد الأقصى: 10MB
                        </small>
                    </div>
                    
                    <button type="submit" class="btn btn-success" style="padding: 10px 20px;">
                        📤 رفع الملف وإضافة الطلاب
                    </button>
                    <button type="button" onclick="document.getElementById('uploadForm').style.display='none'" class="btn btn-secondary">إلغاء</button>
                </form>
                
                <div style="background: #f5f5f5; padding: 15px; border-radius: 5px; margin-top: 15px;">
                    <h4 style="margin-top: 0;">📋 تنسيق الملف المطلوب:</h4>
                    <p><strong>الحقول المطلوبة:</strong></p>
                    <ul style="margin-right: 20px;">
                        <li><strong>student_number</strong> أو <strong>رقم_جامعي</strong> - الرقم الجامعي (مطلوب)</li>
                        <li><strong>full_name_ar</strong> أو <strong>اسم_عربي</strong> - الاسم الكامل بالعربية (مطلوب)</li>
                        <li><strong>full_name_en</strong> أو <strong>اسم_إنجليزي</strong> - الاسم الكامل بالإنجليزية (اختياري)</li>
                        <li><strong>national_id</strong> أو <strong>رقم_وطني</strong> - الرقم الوطني (اختياري)</li>
                        <li><strong>email</strong> أو <strong>بريد</strong> - البريد الإلكتروني (اختياري)</li>
                        <li><strong>phone</strong> أو <strong>هاتف</strong> - رقم الهاتف (اختياري)</li>
                        <li><strong>major_code</strong> أو <strong>رمز_تخصص</strong> - رمز التخصص (اختياري)</li>
                        <li><strong>major_id</strong> - معرف التخصص (اختياري)</li>
                        <li><strong>enrollment_date</strong> أو <strong>تاريخ_تسجيل</strong> - تاريخ التسجيل (اختياري، افتراضي: اليوم)</li>
                    </ul>
                    <p style="margin-top: 10px;"><strong>مثال CSV:</strong></p>
                    <pre style="background: white; padding: 10px; border-radius: 4px; overflow-x: auto;">student_number,full_name_ar,full_name_en,national_id,email,phone,major_code
2021001,أحمد محمد علي,Ahmed Mohammed Ali,123456789012,ahmed@example.com,0501234567,CS001
2021002,فاطمة أحمد حسن,Fatima Ahmed Hassan,234567890123,fatima@example.com,0502345678,CS001
2021003,محمد خالد إبراهيم,Mohammed Khalid Ibrahim,345678901234,mohammed@example.com,0503456789,CS002</pre>
                    <p style="margin-top: 10px;">
                        <a href="sample_students_import.csv" download class="btn btn-secondary" style="padding: 8px 15px; margin-left: 10px;">
                            📥 تحميل ملف نموذجي (CSV)
                        </a>
                    </p>
                    <p style="margin-top: 10px; color: #666; font-size: 14px;">
                        <strong>ملاحظات:</strong>
                        <ul style="margin-right: 20px; margin-top: 5px;">
                            <li>يمكن استخدام أسماء الأعمدة بالعربية أو الإنجليزية</li>
                            <li>الحقول المطلوبة فقط: <code>student_number</code> و <code>full_name_ar</code></li>
                            <li>إذا لم يتم تحديد التخصص في الملف، سيتم استخدام التخصص المحدد في النموذج</li>
                            <li>الطلاب المكررون (بنفس الرقم الجامعي) سيتم تخطيهم تلقائياً</li>
                        </ul>
                    </p>
                </div>
            </div>
        </div>
        
        <div id="addForm" class="card" style="display:none; margin-bottom: 20px;">
            <div class="card-header">
                <h3>إضافة طالب جديد</h3>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="add">
                <div class="form-group">
                    <label>الرقم الجامعي</label>
                    <input type="text" name="student_number" required>
                </div>
                <div class="form-group">
                    <label>الرقم الوطني</label>
                    <input type="text" name="national_id">
                </div>
                <div class="form-group">
                    <label>الاسم الكامل (عربي)</label>
                    <input type="text" name="full_name_ar" required>
                </div>
                <div class="form-group">
                    <label>الاسم الكامل (إنجليزي)</label>
                    <input type="text" name="full_name_en">
                </div>
                <div class="form-group">
                    <label>البريد الإلكتروني</label>
                    <input type="email" name="email">
                </div>
                <div class="form-group">
                    <label>الهاتف</label>
                    <input type="text" name="phone">
                </div>
                <?php if ($auth->hasRole('admin')): ?>
                <div class="form-group">
                    <label>الكلية</label>
                    <select name="college_id" id="add_college_id" required onchange="loadMajorsForAdd()">
                        <option value="">-- اختر الكلية --</option>
                        <?php foreach ($colleges as $college): ?>
                            <option value="<?php echo $college['id']; ?>"><?php echo htmlspecialchars($college['name_ar']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php else: ?>
                    <input type="hidden" name="college_id" id="add_college_id" value="<?php echo $_SESSION['college_id']; ?>">
                <?php endif; ?>
                <div class="form-group">
                    <label>التخصص</label>
                    <select name="major_id" id="add_major_id" required disabled>
                        <option value=""><?php echo $auth->hasRole('admin') ? 'اختر الكلية أولاً' : 'اختر التخصص'; ?></option>
                        <?php foreach ($majors as $major): ?>
                            <option value="<?php echo $major['id']; ?>"><?php echo htmlspecialchars($major['name_ar']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>تاريخ التسجيل</label>
                    <input type="date" name="enrollment_date" value="<?php echo date('Y-m-d'); ?>" required>
                </div>
                <button type="submit" class="btn btn-success">إضافة</button>
                <button type="button" onclick="document.getElementById('addForm').style.display='none'" class="btn btn-secondary">إلغاء</button>
            </form>
        </div>
        
        <!-- قسم حذف جميع الطلاب -->
        <?php if ($auth->hasRole('admin')): ?>
        <div id="deleteAllForm" class="card" style="display:none; margin-bottom: 20px; border: 3px solid #f44336;">
            <div class="card-header" style="background-color: #f44336; color: white;">
                <h3>⚠️ حذف جميع الطلاب</h3>
            </div>
            <div style="padding: 20px;">
        <?php
        // جلب عدد الطلاب للعرض في النموذج
        try {
            $total_students = $db->fetchOne("SELECT COUNT(*) as count FROM students")['count'];
        } catch (Exception $e) {
            $total_students = 0;
        }
        
        // تعريف المتغير للاستخدام في JavaScript
        $js_total_students = $total_students;
        ?>
                <div style="background: #fff3cd; padding: 15px; border-radius: 5px; margin-bottom: 20px; border: 2px solid #ffc107;">
                    <h4 style="color: #856404; margin-top: 0;">⚠️ تحذير خطير!</h4>
                    <p style="color: #856404; font-weight: bold; margin-bottom: 10px;">
                        هذه العملية ستحذف <strong>جميع الطلاب</strong> من النظام بشكل نهائي ولا يمكن التراجع عنها!
                    </p>
                    <p style="color: #856404; margin-bottom: 5px;"><strong>عدد الطلاب الحالي: <?php echo number_format($total_students); ?> طالب</strong></p>
                    <p style="color: #856404; margin-bottom: 0;">
                        <strong>سيتم حذف:</strong>
                    </p>
                    <ul style="color: #856404; margin-right: 20px; margin-top: 10px;">
                        <li>جميع بيانات الطلاب</li>
                        <li>جميع حسابات المستخدمين المرتبطة بالطلاب</li>
                        <li>جميع تسجيلات الطلاب على المواد</li>
                        <li>جميع الدرجات</li>
                        <li>جميع المعدلات التراكمية والفصلية</li>
                        <li>جميع البيانات المرتبطة الأخرى</li>
                    </ul>
                </div>
                
                <form method="POST" onsubmit="return confirmDeleteAll()">
                    <input type="hidden" name="action" value="delete_all">
                    
                    <div class="form-group">
                        <label style="color: #f44336; font-weight: bold; font-size: 16px;">
                            للتأكيد، يرجى كتابة: <code style="background: #f4f4f4; padding: 5px 10px; border-radius: 3px;">حذف جميع الطلاب</code>
                        </label>
                        <input type="text" 
                               name="confirm_text" 
                               id="confirmDeleteText"
                               placeholder="اكتب: حذف جميع الطلاب"
                               required
                               style="width: 100%; max-width: 400px; padding: 12px; border: 2px solid #f44336; border-radius: 5px; font-size: 16px;"
                               oninput="checkConfirmText()">
                        <small style="display: block; margin-top: 5px; color: #666;">
                            يجب كتابة النص بالضبط كما هو موضح أعلاه
                        </small>
                    </div>
                    
                    <div style="margin-top: 20px;">
                        <button type="submit" 
                                id="deleteAllBtn"
                                class="btn btn-danger" 
                                disabled
                                style="padding: 12px 30px; font-size: 16px; font-weight: bold;">
                            🗑️ حذف جميع الطلاب (<?php echo number_format($total_students); ?>)
                        </button>
                        <button type="button" 
                                onclick="document.getElementById('deleteAllForm').style.display='none'; document.getElementById('confirmDeleteText').value=''; checkConfirmText();" 
                                class="btn btn-secondary">
                            إلغاء
                        </button>
                    </div>
                </form>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- قسم البحث -->
        <div class="card" style="margin-bottom: 20px;">
            <div class="card-header">
                <h3>🔍 البحث عن طالب</h3>
            </div>
            <div style="padding: 20px;">
                <div class="form-group" style="position: relative; display: inline-block; width: 100%; max-width: 600px;">
                    <label for="searchInput" style="display: block; margin-bottom: 8px; font-weight: bold;">
                        ابحث عن طالب:
                    </label>
                    <div style="position: relative; display: flex; align-items: center;">
                        <input type="text" 
                               id="searchInput" 
                               placeholder="ابحث بالرقم الجامعي، الاسم، الرقم الوطني، البريد الإلكتروني، الكلية، أو التخصص..."
                               style="padding: 10px 40px 10px 10px; width: 100%; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
                               onkeyup="filterStudents()">
                        <button type="button" 
                                id="clearSearchBtn" 
                                onclick="clearSearch()" 
                                style="position: absolute; left: 10px; background: #f44336; color: white; border: none; border-radius: 50%; width: 28px; height: 28px; cursor: pointer; display: none; font-size: 16px; line-height: 1;"
                                title="مسح البحث">
                            ×
                        </button>
                    </div>
                    <small style="display: block; margin-top: 5px; color: #666;">
                        يمكنك البحث في جميع الحقول: الرقم الجامعي، الاسم، الرقم الوطني، البريد الإلكتروني، الكلية، أو التخصص
                    </small>
                </div>
                <div id="searchResults" style="margin-top: 10px; color: #666; font-size: 14px;"></div>
            </div>
        </div>
        
        <div class="table-container">
            <table id="studentsTable">
                <thead>
                    <tr>
                        <th>الرقم الجامعي</th>
                        <th>الاسم</th>
                        <th>الكلية</th>
                        <th>التخصص</th>
                        <th>الفصل الدراسي</th>
                        <th>السنة الأكاديمية</th>
                        <th>الحالة</th>
                        <th>الإجراءات</th>
                    </tr>
                </thead>
                <tbody id="studentsTableBody">
                    <?php foreach ($students as $student): ?>
                        <tr data-student-number="<?php echo htmlspecialchars($student['student_number']); ?>"
                            data-student-name-ar="<?php echo htmlspecialchars($student['full_name_ar']); ?>"
                            data-student-name-en="<?php echo htmlspecialchars($student['full_name_en'] ?? ''); ?>"
                            data-national-id="<?php echo htmlspecialchars($student['national_id'] ?? ''); ?>"
                            data-email="<?php echo htmlspecialchars($student['email'] ?? ''); ?>"
                            data-college-name="<?php echo htmlspecialchars($student['college_name']); ?>"
                            data-major-name="<?php echo htmlspecialchars($student['major_name']); ?>">
                            <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                            <td><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                            <td><?php echo htmlspecialchars($student['college_name']); ?></td>
                            <td><?php echo htmlspecialchars($student['major_name']); ?></td>
                            <td><?php 
                                if (!empty($student['semester_number'])) {
                                    echo 'فصل ' . htmlspecialchars($student['semester_number']);
                                } else {
                                    echo '-';
                                }
                            ?></td>
                            <td><?php 
                                if (!empty($student['academic_year'])) {
                                    echo htmlspecialchars($student['academic_year']);
                                } else {
                                    echo '-';
                                }
                            ?></td>
                            <td><?php 
                                $status = ['active' => 'نشط', 'graduated' => 'متخرج', 'suspended' => 'معلق', 'withdrawn' => 'منسحب'];
                                echo $status[$student['status']] ?? $student['status'];
                            ?></td>
                            <td>
                                <a href="student_details.php?id=<?php echo $student['id']; ?>" class="btn btn-primary">التفاصيل</a>
                                <a href="grades.php?student_id=<?php echo $student['id']; ?>" class="btn btn-success">الدرجات</a>
                                <button onclick="deleteStudent(<?php echo $student['id']; ?>, '<?php echo htmlspecialchars($student['full_name_ar'], ENT_QUOTES); ?>', '<?php echo htmlspecialchars($student['student_number'], ENT_QUOTES); ?>')" class="btn btn-danger">حذف</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <script>
    // دالة البحث والتصفية
    function filterStudents() {
        const searchInput = document.getElementById('searchInput');
        const clearBtn = document.getElementById('clearSearchBtn');
        const searchTerm = searchInput.value.toLowerCase().trim();
        const tableBody = document.getElementById('studentsTableBody');
        const rows = tableBody.getElementsByTagName('tr');
        const resultsDiv = document.getElementById('searchResults');
        
        // إظهار/إخفاء زر المسح
        if (searchTerm) {
            clearBtn.style.display = 'block';
        } else {
            clearBtn.style.display = 'none';
        }
        
        let visibleCount = 0;
        
        for (let i = 0; i < rows.length; i++) {
            const row = rows[i];
            const studentNumber = (row.getAttribute('data-student-number') || '').toLowerCase();
            const studentNameAr = (row.getAttribute('data-student-name-ar') || '').toLowerCase();
            const studentNameEn = (row.getAttribute('data-student-name-en') || '').toLowerCase();
            const nationalId = (row.getAttribute('data-national-id') || '').toLowerCase();
            const email = (row.getAttribute('data-email') || '').toLowerCase();
            const collegeName = (row.getAttribute('data-college-name') || '').toLowerCase();
            const majorName = (row.getAttribute('data-major-name') || '').toLowerCase();
            
            // البحث في جميع الحقول
            const matches = !searchTerm || 
                studentNumber.includes(searchTerm) ||
                studentNameAr.includes(searchTerm) ||
                studentNameEn.includes(searchTerm) ||
                nationalId.includes(searchTerm) ||
                email.includes(searchTerm) ||
                collegeName.includes(searchTerm) ||
                majorName.includes(searchTerm);
            
            if (matches) {
                row.style.display = '';
                visibleCount++;
            } else {
                row.style.display = 'none';
            }
        }
        
        // عرض عدد النتائج
        if (searchTerm) {
            resultsDiv.innerHTML = `تم العثور على <strong>${visibleCount}</strong> طالب`;
            if (visibleCount === 0) {
                resultsDiv.innerHTML = '<span style="color: #d32f2f;">لم يتم العثور على أي نتائج</span>';
            }
        } else {
            resultsDiv.innerHTML = '';
        }
    }
    
    // دالة مسح البحث
    function clearSearch() {
        const searchInput = document.getElementById('searchInput');
        searchInput.value = '';
        filterStudents();
        searchInput.focus();
    }
    
    // دالة لتحديث قائمة التخصصات في نموذج رفع Excel
    function loadMajorsForUpload() {
        const collegeId = document.getElementById('upload_college_id').value;
        const majorSelect = document.getElementById('upload_major_id');
        
        if (!collegeId) {
            majorSelect.innerHTML = '<option value="">-- اختر الكلية أولاً --</option>';
            majorSelect.disabled = true;
            return;
        }
        
        majorSelect.innerHTML = '<option value="">-- اختر التخصص (اختياري) --</option>';
        majorSelect.disabled = false;
        
        fetch(`students.php?action=get_majors&college_id=${collegeId}`)
            .then(response => response.json())
            .then(data => {
                data.forEach(major => {
                    const option = document.createElement('option');
                    option.value = major.id;
                    option.textContent = major.name_ar;
                    majorSelect.appendChild(option);
                });
                majorSelect.disabled = false;
            })
            .catch(error => {
                console.error('Error loading majors:', error);
                majorSelect.innerHTML = '<option value="">خطأ في تحميل التخصصات</option>';
            });
    }
    
    // دالة لتحديث قائمة التخصصات في نموذج إضافة طالب
    function loadMajorsForAdd() {
        const collegeId = document.getElementById('add_college_id').value;
        const majorSelect = document.getElementById('add_major_id');
        
        if (!collegeId) {
            majorSelect.innerHTML = '<option value="">اختر الكلية أولاً</option>';
            majorSelect.disabled = true;
            return;
        }
        
        majorSelect.innerHTML = '<option value="">اختر التخصص</option>';
        majorSelect.disabled = false;
        
        fetch(`students.php?action=get_majors&college_id=${collegeId}`)
            .then(response => response.json())
            .then(data => {
                data.forEach(major => {
                    const option = document.createElement('option');
                    option.value = major.id;
                    option.textContent = major.name_ar;
                    majorSelect.appendChild(option);
                });
                majorSelect.disabled = false;
            })
            .catch(error => {
                console.error('Error loading majors:', error);
                majorSelect.innerHTML = '<option value="">خطأ في تحميل التخصصات</option>';
            });
    }
    
    // تحميل التخصصات عند فتح النموذج إذا كانت الكلية محددة
    document.addEventListener('DOMContentLoaded', function() {
        <?php if ($auth->hasRole('admin')): ?>
        // إذا كان admin، نتحقق من وجود كلية محددة مسبقاً
        const uploadCollegeId = document.getElementById('upload_college_id');
        const addCollegeId = document.getElementById('add_college_id');
        
        if (uploadCollegeId && uploadCollegeId.value) {
            loadMajorsForUpload();
        }
        if (addCollegeId && addCollegeId.value) {
            loadMajorsForAdd();
        }
        <?php else: ?>
        // إذا لم يكن admin، الكلية محددة مسبقاً، نحمل التخصصات
        const uploadCollegeId = document.getElementById('upload_college_id');
        const addCollegeId = document.getElementById('add_college_id');
        
        if (uploadCollegeId && uploadCollegeId.value) {
            loadMajorsForUpload();
        }
        if (addCollegeId && addCollegeId.value) {
            loadMajorsForAdd();
        }
        <?php endif; ?>
    });
    
    function deleteStudent(studentId, studentName, studentNumber) {
        if (confirm('هل أنت متأكد من حذف الطالب:\nالاسم: ' + studentName + '\nالرقم الجامعي: ' + studentNumber + '\n\nسيتم حذف جميع البيانات المرتبطة بهذا الطالب (الدرجات، التسجيلات، إلخ).\nهذا الإجراء لا يمكن التراجع عنه!')) {
            // إنشاء نموذج مخفي لإرسال طلب الحذف
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = 'students.php';
            
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'delete';
            form.appendChild(actionInput);
            
            const studentIdInput = document.createElement('input');
            studentIdInput.type = 'hidden';
            studentIdInput.name = 'student_id';
            studentIdInput.value = studentId;
            form.appendChild(studentIdInput);
            
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // دالة للتحقق من نص التأكيد لحذف جميع الطلاب
    function checkConfirmText() {
        const confirmText = document.getElementById('confirmDeleteText');
        const deleteBtn = document.getElementById('deleteAllBtn');
        const expectedText = 'حذف جميع الطلاب';
        
        if (confirmText && deleteBtn) {
            if (confirmText.value.trim() === expectedText) {
                deleteBtn.disabled = false;
                deleteBtn.style.opacity = '1';
            } else {
                deleteBtn.disabled = true;
                deleteBtn.style.opacity = '0.5';
            }
        }
    }
    
    // دالة تأكيد نهائية قبل الحذف
    function confirmDeleteAll() {
        const confirmText = document.getElementById('confirmDeleteText');
        const expectedText = 'حذف جميع الطلاب';
        
        if (!confirmText || confirmText.value.trim() !== expectedText) {
            alert('⚠️ يجب كتابة "حذف جميع الطلاب" بالضبط للتأكيد');
            return false;
        }
        
        <?php
        // جلب عدد الطلاب للاستخدام في JavaScript
        try {
            $js_total_students = $db->fetchOne("SELECT COUNT(*) as count FROM students")['count'];
        } catch (Exception $e) {
            $js_total_students = 0;
        }
        ?>
        const totalStudents = <?php echo $js_total_students; ?>;
        
        if (totalStudents === 0) {
            alert('لا يوجد طلاب لحذفهم');
            return false;
        }
        
        const finalConfirm = confirm(
            '⚠️⚠️⚠️ تحذير نهائي ⚠️⚠️⚠️\n\n' +
            'أنت على وشك حذف جميع الطلاب (' + totalStudents + ' طالب) من النظام!\n\n' +
            'هذه العملية:\n' +
            '• لا يمكن التراجع عنها\n' +
            '• ستحذف جميع بيانات الطلاب\n' +
            '• ستحذف جميع الدرجات والتسجيلات\n' +
            '• ستحذف جميع المعدلات\n' +
            '• ستحذف جميع البيانات المرتبطة\n\n' +
            'هل أنت متأكد تماماً من رغبتك في المتابعة؟'
        );
        
        if (!finalConfirm) {
            return false;
        }
        
        // تأكيد إضافي
        const secondConfirm = confirm(
            '⚠️ تأكيد نهائي أخير ⚠️\n\n' +
            'أنت على وشك حذف ' + totalStudents + ' طالب بشكل نهائي!\n\n' +
            'اضغط "موافق" فقط إذا كنت متأكداً 100%'
        );
        
        return secondConfirm;
    }
    </script>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

