<?php
/**
 * صفحة التقارير الإحصائية
 * Statistical Reports Page
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin') && !$auth->hasRole('registrar'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();

// معالجة طلبات AJAX
if (isset($_GET['action'])) {
    header('Content-Type: application/json');
    if ($_GET['action'] === 'get_majors') {
        $college_id_ajax = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
        if ($college_id_ajax) {
            if ($auth->hasRole('admin') || ($auth->belongsToCollege($college_id_ajax))) {
                $majors_ajax = $db->fetchAll(
                    "SELECT * FROM majors WHERE college_id = ? ORDER BY name_ar",
                    [$college_id_ajax]
                );
                echo json_encode($majors_ajax);
            } else {
                echo json_encode([]);
            }
        } else {
            echo json_encode([]);
        }
        exit;
    } elseif ($_GET['action'] === 'get_semesters') {
        $major_id_ajax = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
        if ($major_id_ajax) {
            $major_info = $db->fetchOne("SELECT * FROM majors WHERE id = ?", [$major_id_ajax]);
            if ($major_info && ($auth->hasRole('admin') || $auth->belongsToCollege($major_info['college_id']))) {
                $semesters_ajax = $db->fetchAll(
                    "SELECT s.*, m.name_ar as major_name, m.college_id 
                     FROM semesters s 
                     JOIN majors m ON s.major_id = m.id 
                     WHERE s.major_id = ?
                     ORDER BY s.academic_year DESC, s.semester_number DESC",
                    [$major_id_ajax]
                );
                echo json_encode($semesters_ajax);
            } else {
                echo json_encode([]);
            }
        } else {
            echo json_encode([]);
        }
        exit;
    }
}

// تحديد الكلية والتخصص والفصل الدراسي
$college_id = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;

// جلب الكليات
$colleges = [];
if ($auth->hasRole('admin')) {
    $colleges = $db->fetchAll("SELECT * FROM colleges ORDER BY name_ar");
} else {
    // لغير admin، تحديد الكلية تلقائياً
    if (!isset($_GET['college_id']) && isset($_SESSION['college_id'])) {
        $college_id = $_SESSION['college_id'];
    }
    $colleges = $db->fetchAll("SELECT * FROM colleges WHERE id = ?", [$_SESSION['college_id']]);
}

// جلب التخصصات حسب الكلية
$majors = [];
if ($college_id) {
    $majors = $db->fetchAll(
        "SELECT * FROM majors WHERE college_id = ? ORDER BY name_ar",
        [$college_id]
    );
}

// جلب الفصول الدراسية حسب التخصص
$semesters = [];
if ($major_id) {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name, m.college_id 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         WHERE s.major_id = ?
         ORDER BY s.academic_year DESC, s.semester_number DESC",
        [$major_id]
    );
}

// جلب البيانات للتقرير الأول (الإحصاءات)
$report1_data = [];
// جلب البيانات للتقرير الثاني (التوزيع)
$report2_data = [];
// جلب البيانات للتقرير الثالث (إحصائيات الطلاب)
$report3_data = [];

if ($semester_id) {
    // جلب جميع المواد في الفصل الدراسي التي لديها درجات
    $courses = $db->fetchAll(
        "SELECT DISTINCT c.id, c.code, c.name_ar 
         FROM courses c
         JOIN student_courses sc ON c.id = sc.course_id
         JOIN grades g ON sc.id = g.student_course_id
         WHERE sc.semester_id = ? AND g.semester_id = ?
         ORDER BY c.code",
        [$semester_id, $semester_id]
    );
    
    foreach ($courses as $course) {
        // جلب جميع الدرجات للمادة
        $grades = $db->fetchAll(
            "SELECT g.marks, g.letter_grade
             FROM grades g
             JOIN student_courses sc ON g.student_course_id = sc.id
             WHERE g.course_id = ? AND g.semester_id = ?",
            [$course['id'], $semester_id]
        );
        
        if (empty($grades)) {
            continue;
        }
        
        // التقرير الأول: الإحصاءات
        $numeric_grades = [];
        foreach ($grades as $grade) {
            $marks = $grade['marks'];
            // التحقق إذا كانت الدرجة رقمية
            if (is_numeric($marks)) {
                $numeric_grades[] = (float)$marks;
            }
        }
        
        $min_grade = !empty($numeric_grades) ? min($numeric_grades) : null;
        $max_grade = !empty($numeric_grades) ? max($numeric_grades) : null;
        $avg_grade = !empty($numeric_grades) ? array_sum($numeric_grades) / count($numeric_grades) : null;
        
        // حساب الانحراف المعياري
        $std_dev = null;
        if (!empty($numeric_grades) && count($numeric_grades) > 1) {
            $variance = 0;
            foreach ($numeric_grades as $grade) {
                $variance += pow($grade - $avg_grade, 2);
            }
            $std_dev = sqrt($variance / count($numeric_grades));
        } elseif (!empty($numeric_grades) && count($numeric_grades) == 1) {
            $std_dev = 0;
        }
        
        $report1_data[] = [
            'course_name' => $course['name_ar'],
            'min_grade' => $min_grade !== null ? number_format($min_grade, 2) : '-',
            'max_grade' => $max_grade !== null ? number_format($max_grade, 2) : '-',
            'avg_grade' => $avg_grade !== null ? number_format($avg_grade, 2) : '-',
            'std_dev' => $std_dev !== null ? number_format($std_dev, 2) : '-'
        ];
        
        // التقرير الثاني: التوزيع
        $total_students = count($grades);
        $excellent = 0;      // امتياز (A: 80-100)
        $very_good = 0;      // جيد (B: 70-79)
        $good = 0;           // مقبول (C: 60-69)
        $acceptable = 0;     // مقبول (D: 50-59)
        $fail = 0;           // رسوب (F: <50)
        $absent = 0;         // غياب
        $alternative = 0;    // بديل
        $cheating = 0;       // غش
        
        foreach ($grades as $grade) {
            $marks = $grade['marks'];
            $letter_grade = trim($grade['letter_grade'] ?? '');
            
            // التحقق من الدرجات النصية أولاً
            $marks_str = $marks !== null ? trim((string)$marks) : '';
            
            // التحقق من الحالات النصية (غياب، غش، بديل)
            if ($marks_str === 'غياب' || $marks_str === 'غ' || $letter_grade === 'غياب') {
                $absent++;
                continue;
            }
            if ($marks_str === 'غش' || $marks_str === 'غشش' || $letter_grade === 'غش') {
                $cheating++;
                continue;
            }
            if ($marks_str === 'بد' || $marks_str === 'ب' || $letter_grade === 'بد') {
                $alternative++;
                continue;
            }
            
            // التحقق من الدرجات الرقمية أو الحروف
            // التقدير الجديد: A (80-100), B (70-79), C (60-69), D (50-59), F (<50)
            if (is_numeric($marks)) {
                $marks_num = (float)$marks;
                if ($marks_num >= 80 && $marks_num <= 100) {
                    $excellent++;
                } elseif ($marks_num >= 70 && $marks_num < 80) {
                    $very_good++;
                } elseif ($marks_num >= 60 && $marks_num < 70) {
                    $good++;
                } elseif ($marks_num >= 50 && $marks_num < 60) {
                    $acceptable++;
                } elseif ($marks_num < 50) {
                    $fail++;
                }
            } elseif ($letter_grade) {
                // استخدام letter_grade إذا كانت marks غير رقمية
                if ($letter_grade === 'A') {
                    $excellent++;
                } elseif ($letter_grade === 'B') {
                    $very_good++;
                } elseif ($letter_grade === 'C') {
                    $good++;
                } elseif ($letter_grade === 'D') {
                    $acceptable++;
                } elseif ($letter_grade === 'F') {
                    $fail++;
                }
            }
        }
        
        $report2_data[] = [
            'course_name' => $course['name_ar'],
            'total_students' => $total_students,
            'excellent' => $excellent,
            'very_good' => $very_good,
            'good' => $good,
            'acceptable' => $acceptable,
            'fail' => $fail,
            'absent' => $absent,
            'alternative' => $alternative,
            'cheating' => $cheating
        ];
    }
    
    // التقرير الثالث: إحصائيات الطلاب
    if ($semester_id) {
        // جلب جميع الطلاب المسجلين في الفصل الدراسي
        $all_students = $db->fetchAll(
            "SELECT DISTINCT s.id, s.student_number, s.full_name_ar, s.status
             FROM students s
             JOIN student_courses sc ON s.id = sc.student_id
             WHERE sc.semester_id = ?",
            [$semester_id]
        );
        
        $total_students_count = count($all_students);
        
        if ($total_students_count > 0) {
            // جلب المعدلات الفصلية للطلاب
            $semester_gpas = $db->fetchAll(
                "SELECT student_id, gpa 
                 FROM semester_gpas 
                 WHERE semester_id = ?",
                [$semester_id]
            );
            
            $gpa_map = [];
            foreach ($semester_gpas as $sgpa) {
                $gpa_map[$sgpa['student_id']] = (float)$sgpa['gpa'];
            }
            
            // جلب الدرجات لجميع الطلاب في الفصل
            $all_grades = $db->fetchAll(
                "SELECT g.student_id, g.marks, g.letter_grade
                 FROM grades g
                 JOIN student_courses sc ON g.student_course_id = sc.id
                 WHERE sc.semester_id = ? AND g.semester_id = ?",
                [$semester_id, $semester_id]
            );
            
            // تجميع البيانات لكل طالب
            $student_data = [];
            foreach ($all_students as $student) {
                $student_id = $student['id'];
                $student_data[$student_id] = [
                    'excellent' => false,
                    'very_good' => false,
                    'good' => false,
                    'acceptable' => false,
                    'alternative' => false,
                    'supplementary' => false,
                    'returning' => false,
                    'cheating' => false,
                    'frozen' => false
                ];
                
                // التحقق من المعدل الفصلي
                if (isset($gpa_map[$student_id])) {
                    $gpa = $gpa_map[$student_id];
                    if ($gpa >= 3.5) {
                        $student_data[$student_id]['excellent'] = true;
                    } elseif ($gpa >= 3.0) {
                        $student_data[$student_id]['very_good'] = true;
                    } elseif ($gpa >= 2.5) {
                        $student_data[$student_id]['good'] = true;
                    } elseif ($gpa >= 2.0) {
                        $student_data[$student_id]['acceptable'] = true;
                    }
                }
                
                // التحقق من حالة الطالب
                if ($student['status'] === 'suspended') {
                    $student_data[$student_id]['frozen'] = true;
                }
                
                // التحقق من الدرجات
                $has_alternative = false;
                $has_supplementary = false;
                $has_cheating = false;
                $fail_count = 0;
                
                foreach ($all_grades as $grade) {
                    if ($grade['student_id'] == $student_id) {
                        $marks = $grade['marks'];
                        $letter_grade = trim($grade['letter_grade'] ?? '');
                        $marks_str = $marks !== null ? trim((string)$marks) : '';
                        
                        // التحقق من البدائل
                        if ($marks_str === 'بد' || $marks_str === 'ب' || $letter_grade === 'بد') {
                            $has_alternative = true;
                        }
                        
                        // التحقق من الغش
                        if ($marks_str === 'غش' || $marks_str === 'غشش' || $letter_grade === 'غش') {
                            $has_cheating = true;
                        }
                        
                        // التحقق من الرسوب (الملاحق)
                        if (is_numeric($marks)) {
                            $marks_num = (float)$marks;
                            if ($marks_num < 50 || $letter_grade === 'F') {
                                $fail_count++;
                                $has_supplementary = true;
                            }
                        } elseif ($letter_grade === 'F') {
                            $fail_count++;
                            $has_supplementary = true;
                        }
                    }
                }
                
                if ($has_alternative) {
                    $student_data[$student_id]['alternative'] = true;
                }
                if ($has_supplementary) {
                    $student_data[$student_id]['supplementary'] = true;
                }
                if ($has_cheating) {
                    $student_data[$student_id]['cheating'] = true;
                }
                if ($fail_count >= 3 || $student['status'] === 'suspended' || $student['status'] === 'withdrawn') {
                    $student_data[$student_id]['returning'] = true;
                }
            }
            
            // حساب الإحصائيات
            $excellent_count = 0;
            $very_good_count = 0;
            $good_count = 0;
            $acceptable_count = 0;
            $alternative_count = 0;
            $supplementary_count = 0;
            $returning_count = 0;
            $cheating_count = 0;
            $frozen_count = 0;
            
            foreach ($student_data as $data) {
                if ($data['excellent']) $excellent_count++;
                if ($data['very_good']) $very_good_count++;
                if ($data['good']) $good_count++;
                if ($data['acceptable']) $acceptable_count++;
                if ($data['alternative']) $alternative_count++;
                if ($data['supplementary']) $supplementary_count++;
                if ($data['returning']) $returning_count++;
                if ($data['cheating']) $cheating_count++;
                if ($data['frozen']) $frozen_count++;
            }
            
            $report3_data = [
                'total_students' => $total_students_count,
                'excellent' => $excellent_count,
                'excellent_percent' => $total_students_count > 0 ? ($excellent_count / $total_students_count) * 100 : 0,
                'very_good' => $very_good_count,
                'very_good_percent' => $total_students_count > 0 ? ($very_good_count / $total_students_count) * 100 : 0,
                'good' => $good_count,
                'good_percent' => $total_students_count > 0 ? ($good_count / $total_students_count) * 100 : 0,
                'acceptable' => $acceptable_count,
                'acceptable_percent' => $total_students_count > 0 ? ($acceptable_count / $total_students_count) * 100 : 0,
                'alternative' => $alternative_count,
                'alternative_percent' => $total_students_count > 0 ? ($alternative_count / $total_students_count) * 100 : 0,
                'supplementary' => $supplementary_count,
                'supplementary_percent' => $total_students_count > 0 ? ($supplementary_count / $total_students_count) * 100 : 0,
                'returning' => $returning_count,
                'returning_percent' => $total_students_count > 0 ? ($returning_count / $total_students_count) * 100 : 0,
                'cheating' => $cheating_count,
                'cheating_percent' => $total_students_count > 0 ? ($cheating_count / $total_students_count) * 100 : 0,
                'frozen' => $frozen_count,
                'frozen_percent' => $total_students_count > 0 ? ($frozen_count / $total_students_count) * 100 : 0
            ];
        }
    }
}

// جلب معلومات الفصل المحدد
$selected_semester = null;
if ($semester_id) {
    $selected_semester = $db->fetchOne(
        "SELECT s.*, m.name_ar as major_name 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         WHERE s.id = ?",
        [$semester_id]
    );
}

?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تقارير النتائج الإحصائية - نظام نتائج جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
</head>
<body>
<?php require_once __DIR__ . '/includes/header.php'; ?>

<div class="container">
    <h1>تقارير النتائج الإحصائية</h1>
    
    <!-- نموذج اختيار الكلية والتخصص والفصل -->
    <div class="form-section">
        <form method="GET" action="reports.php" id="reportsForm" class="form-inline">
            <div class="form-group">
                <label for="college_id">اختر الكلية:</label>
                <select name="college_id" id="college_id" required onchange="loadMajors()" <?php echo (!$auth->hasRole('admin') && count($colleges) == 1) ? 'disabled' : ''; ?>>
                    <?php if (count($colleges) > 1 || $auth->hasRole('admin')): ?>
                        <option value="">-- اختر الكلية --</option>
                    <?php endif; ?>
                    <?php foreach ($colleges as $college): ?>
                        <option value="<?php echo $college['id']; ?>" <?php echo ($college_id == $college['id'] || (!$auth->hasRole('admin') && count($colleges) == 1)) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($college['name_ar']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <?php if (!$auth->hasRole('admin') && count($colleges) == 1): ?>
                    <input type="hidden" name="college_id" value="<?php echo $college_id; ?>">
                <?php endif; ?>
            </div>
            
            <div class="form-group">
                <label for="major_id">اختر التخصص:</label>
                <select name="major_id" id="major_id" required onchange="loadSemesters()" <?php echo !$college_id ? 'disabled' : ''; ?>>
                    <option value="">-- اختر التخصص --</option>
                    <?php foreach ($majors as $major): ?>
                        <option value="<?php echo $major['id']; ?>" <?php echo $major_id == $major['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($major['name_ar']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-group">
                <label for="semester_id">اختر الفصل الدراسي:</label>
                <select name="semester_id" id="semester_id" required <?php echo !$major_id ? 'disabled' : ''; ?>>
                    <option value="">-- اختر الفصل --</option>
                    <?php foreach ($semesters as $sem): ?>
                        <option value="<?php echo $sem['id']; ?>" <?php echo $semester_id == $sem['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($sem['academic_year'] . ' - فصل ' . $sem['semester_number']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-group">
                <button type="submit" class="btn btn-primary" id="submitBtn" <?php echo !$semester_id ? 'disabled' : ''; ?>>عرض التقارير</button>
            </div>
        </form>
    </div>
    
    <?php if ($semester_id && $selected_semester): ?>
        <div class="semester-info">
            <h2>الفصل الدراسي: <?php echo htmlspecialchars($selected_semester['major_name'] . ' - ' . $selected_semester['academic_year'] . ' - فصل ' . $selected_semester['semester_number']); ?></h2>
        </div>
        
        <!-- التقرير الأول: الإحصاءات -->
        <div class="report-section">
            <h2>التقرير الأول: الإحصاءات الوصفية للمواد</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>اسم المادة</th>
                            <th>أصغر درجة</th>
                            <th>أكبر درجة</th>
                            <th>المتوسط</th>
                            <th>الانحراف المعياري</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($report1_data)): ?>
                            <tr>
                                <td colspan="5" class="text-center">لا توجد بيانات متاحة</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($report1_data as $row): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($row['course_name']); ?></td>
                                    <td><?php echo $row['min_grade']; ?></td>
                                    <td><?php echo $row['max_grade']; ?></td>
                                    <td><?php echo $row['avg_grade']; ?></td>
                                    <td><?php echo $row['std_dev']; ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <!-- الرسم البياني للتقرير الأول -->
            <div class="chart-container">
                <canvas id="chart1"></canvas>
            </div>
        </div>
        
        <!-- التقرير الثاني: التوزيع -->
        <div class="report-section">
            <h2>التقرير الثاني: توزيع الدرجات حسب المادة</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>اسم المادة</th>
                            <th>عدد الجالسين</th>
                            <th>امتياز</th>
                            <th>جيد جدا</th>
                            <th>جيد</th>
                            <th>مقبول</th>
                            <th>رسوب</th>
                            <th>غياب</th>
                            <th>بديل</th>
                            <th>غش</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($report2_data)): ?>
                            <tr>
                                <td colspan="10" class="text-center">لا توجد بيانات متاحة</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($report2_data as $row): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($row['course_name']); ?></td>
                                    <td><?php echo $row['total_students']; ?></td>
                                    <td><?php echo $row['excellent']; ?></td>
                                    <td><?php echo $row['very_good']; ?></td>
                                    <td><?php echo $row['good']; ?></td>
                                    <td><?php echo $row['acceptable']; ?></td>
                                    <td><?php echo $row['fail']; ?></td>
                                    <td><?php echo $row['absent']; ?></td>
                                    <td><?php echo $row['alternative']; ?></td>
                                    <td><?php echo $row['cheating']; ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <!-- الرسم البياني للتقرير الثاني -->
            <div class="chart-container">
                <canvas id="chart2"></canvas>
            </div>
        </div>
        
        <!-- التقرير الثالث: إحصائيات الطلاب -->
        <div class="report-section">
            <h2>التقرير الثالث: إحصائيات الطلاب حسب التقديرات والحالات</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>التصنيف</th>
                            <th>العدد</th>
                            <th>النسبة المئوية</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($report3_data)): ?>
                            <tr>
                                <td colspan="3" class="text-center">لا توجد بيانات متاحة</td>
                            </tr>
                        <?php else: ?>
                            <tr>
                                <td><strong>ممتاز</strong></td>
                                <td><?php echo $report3_data['excellent']; ?></td>
                                <td><?php echo number_format($report3_data['excellent_percent'], 2); ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>جيد جدا</strong></td>
                                <td><?php echo $report3_data['very_good']; ?></td>
                                <td><?php echo number_format($report3_data['very_good_percent'], 2); ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>جيد</strong></td>
                                <td><?php echo $report3_data['good']; ?></td>
                                <td><?php echo number_format($report3_data['good_percent'], 2); ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>مقبول</strong></td>
                                <td><?php echo $report3_data['acceptable']; ?></td>
                                <td><?php echo number_format($report3_data['acceptable_percent'], 2); ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>امتحانات البدائل</strong></td>
                                <td><?php echo $report3_data['alternative']; ?></td>
                                <td><?php echo number_format($report3_data['alternative_percent'], 2); ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>امتحانات الملاحق</strong></td>
                                <td><?php echo $report3_data['supplementary']; ?></td>
                                <td><?php echo number_format($report3_data['supplementary_percent'], 2); ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>العائدين للفصل الدراسي</strong></td>
                                <td><?php echo $report3_data['returning']; ?></td>
                                <td><?php echo number_format($report3_data['returning_percent'], 2); ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>طلاب الغش</strong></td>
                                <td><?php echo $report3_data['cheating']; ?></td>
                                <td><?php echo number_format($report3_data['cheating_percent'], 2); ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>طلاب التجميد</strong></td>
                                <td><?php echo $report3_data['frozen']; ?></td>
                                <td><?php echo number_format($report3_data['frozen_percent'], 2); ?>%</td>
                            </tr>
                            <tr style="background-color: #e8f4f8; font-weight: bold;">
                                <td><strong>إجمالي الطلاب</strong></td>
                                <td><?php echo $report3_data['total_students']; ?></td>
                                <td>100.00%</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <!-- الرسم البياني للتقرير الثالث -->
            <div class="chart-container">
                <canvas id="chart3"></canvas>
            </div>
        </div>
        
        <!-- أزرار التصدير -->
        <div class="export-buttons">
            <button onclick="exportToExcel('report1')" class="btn btn-success">تصدير التقرير الأول إلى Excel</button>
            <button onclick="exportToExcel('report2')" class="btn btn-success">تصدير التقرير الثاني إلى Excel</button>
            <button onclick="exportToExcel('report3')" class="btn btn-success">تصدير التقرير الثالث إلى Excel</button>
        </div>
    <?php endif; ?>
</div>

<style>
/* تحسين التصميم العام */
.container h1 {
    color: #2c3e50;
    margin-bottom: 30px;
    font-size: 2em;
    text-align: center;
    padding-bottom: 15px;
    border-bottom: 3px solid #667eea;
    direction: rtl;
}

.form-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    margin-bottom: 30px;
    direction: rtl;
}

.report-section {
    margin-top: 30px;
    margin-bottom: 40px;
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    direction: rtl;
}

.report-section h2 {
    color: #2c3e50;
    border-bottom: 3px solid #667eea;
    padding-bottom: 15px;
    margin-bottom: 25px;
    font-size: 1.5em;
    text-align: right;
}

.semester-info {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 20px;
    border-radius: 10px;
    margin: 20px 0;
    box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
    direction: rtl;
    text-align: right;
}

.semester-info h2 {
    margin: 0;
    color: white;
    font-size: 1.3em;
    text-align: right;
}

.export-buttons {
    margin-top: 30px;
    text-align: center;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 10px;
    direction: rtl;
}

.export-buttons button {
    margin: 5px 10px;
    padding: 12px 25px;
    font-size: 1em;
    border-radius: 5px;
    transition: all 0.3s;
}

.export-buttons button:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
}

.form-inline {
    display: flex;
    flex-wrap: wrap;
    gap: 20px;
    align-items: flex-end;
}

.form-inline .form-group {
    flex: 1;
    min-width: 200px;
}

.form-inline label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #333;
    font-size: 1em;
}

.form-inline select {
    width: 100%;
    padding: 12px;
    border: 2px solid #ddd;
    border-radius: 5px;
    font-size: 1em;
    transition: border-color 0.3s;
}

.form-inline select:focus {
    outline: none;
    border-color: #667eea;
}

.form-inline select:disabled {
    background-color: #f5f5f5;
    cursor: not-allowed;
    opacity: 0.7;
}

.form-inline button {
    padding: 12px 30px;
    font-size: 1em;
    border-radius: 5px;
    transition: all 0.3s;
}

.form-inline button:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

.form-inline button:not(:disabled):hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
}

/* تنسيقات الجداول */
.table-responsive {
    overflow-x: auto;
    margin: 20px 0;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    direction: rtl;
}

.data-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
    font-size: 1em;
    direction: rtl;
}

.data-table thead {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.data-table th {
    padding: 15px;
    text-align: right;
    font-weight: 600;
    font-size: 1em;
    border: none;
    white-space: nowrap;
}

.data-table th:first-child {
    border-top-left-radius: 8px;
}

.data-table th:last-child {
    border-top-right-radius: 8px;
}

.data-table td {
    padding: 12px 15px;
    text-align: right;
    border-bottom: 1px solid #e0e0e0;
    color: #333;
}

.data-table tbody tr {
    transition: background-color 0.2s;
}

.data-table tbody tr:hover {
    background-color: #f8f9fa;
}

.data-table tbody tr:last-child td {
    border-bottom: none;
}

.data-table tbody tr:nth-child(even) {
    background-color: #fafafa;
}

.data-table tbody tr:nth-child(even):hover {
    background-color: #f0f0f0;
}

.text-center {
    text-align: center;
    padding: 20px;
    color: #666;
    font-style: italic;
}

/* تحسين الأزرار */
.btn {
    padding: 12px 25px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 1em;
    font-weight: 600;
    transition: all 0.3s;
    text-decoration: none;
    display: inline-block;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.btn-primary:hover:not(:disabled) {
    background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
}

.btn-success {
    background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
    color: white;
}

.btn-success:hover {
    background: linear-gradient(135deg, #45a049 0%, #4CAF50 100%);
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(76, 175, 80, 0.3);
}

/* تصميم متجاوب */
@media (max-width: 768px) {
    .form-inline {
        flex-direction: column;
    }
    
    .form-inline .form-group {
        width: 100%;
    }
    
    .data-table {
        font-size: 0.9em;
    }
    
    .data-table th,
    .data-table td {
        padding: 8px;
    }
    
    .export-buttons {
        display: flex;
        flex-direction: column;
        gap: 10px;
    }
    
    .export-buttons button {
        width: 100%;
        margin: 0;
    }
}

/* تنسيقات الرسوم البيانية */
canvas {
    max-height: 400px;
    margin: 20px auto;
    display: block;
}

.chart-container {
    position: relative;
    height: 400px;
    width: 100%;
    margin: 30px 0;
    padding: 20px;
    background: #fafafa;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}
</style>

<script>
// تحميل التخصصات والفصول عند تحميل الصفحة إذا كانت محددة مسبقاً
document.addEventListener('DOMContentLoaded', function() {
    const collegeSelect = document.getElementById('college_id');
    const majorSelect = document.getElementById('major_id');
    const semesterSelect = document.getElementById('semester_id');
    const submitButton = document.getElementById('submitBtn');
    
    if (!collegeSelect || !majorSelect || !semesterSelect || !submitButton) {
        console.error('Required elements not found');
        return;
    }
    
    const collegeId = collegeSelect.value;
    const majorId = majorSelect.value;
    
    // إضافة event listener للفصل الدراسي لتفعيل الزر
    semesterSelect.addEventListener('change', function() {
        submitButton.disabled = !this.value;
    });
    
    // إذا كانت الكلية محددة، تحميل التخصصات تلقائياً
    if (collegeId) {
        if (!majorId) {
            // تحميل التخصصات فقط
            loadMajors();
        } else {
            // تحميل التخصصات والفصول
            loadMajors();
            setTimeout(function() {
                const majorSelectEl = document.getElementById('major_id');
                if (majorSelectEl) {
                    majorSelectEl.value = majorId;
                    loadSemesters();
                }
            }, 500);
        }
    }
});

function loadMajors() {
    const collegeSelect = document.getElementById('college_id');
    const majorSelect = document.getElementById('major_id');
    const semesterSelect = document.getElementById('semester_id');
    const submitButton = document.getElementById('submitBtn');
    
    if (!collegeSelect || !majorSelect || !semesterSelect || !submitButton) {
        console.error('Required elements not found in loadMajors');
        return;
    }
    
    const collegeId = collegeSelect.value;
    const currentMajorId = majorSelect.value;
    
    // إعادة تعيين التخصصات والفصول
    majorSelect.innerHTML = '<option value="">-- اختر التخصص --</option>';
    semesterSelect.innerHTML = '<option value="">-- اختر الفصل --</option>';
    majorSelect.disabled = !collegeId;
    semesterSelect.disabled = true;
    submitButton.disabled = true;
    
    if (!collegeId) {
        return;
    }
    
    // جلب التخصصات عبر AJAX
    fetch(`reports.php?action=get_majors&college_id=${collegeId}`)
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok: ' + response.status);
            }
            return response.json();
        })
        .then(data => {
            if (!Array.isArray(data)) {
                console.error('Invalid data format:', data);
                alert('خطأ في البيانات المستلمة');
                return;
            }
            
            if (data.length === 0) {
                alert('لا توجد تخصصات متاحة لهذه الكلية');
                return;
            }
            
            data.forEach(major => {
                const option = document.createElement('option');
                option.value = major.id;
                option.textContent = major.name_ar;
                if (major.id == currentMajorId) {
                    option.selected = true;
                }
                majorSelect.appendChild(option);
            });
            majorSelect.disabled = false;
            
            // إذا كان هناك تخصص محدد مسبقاً، تحميل الفصول
            if (currentMajorId) {
                setTimeout(function() {
                    loadSemesters();
                }, 100);
            }
        })
        .catch(error => {
            console.error('Error loading majors:', error);
            alert('حدث خطأ في تحميل التخصصات: ' + error.message);
        });
}

function loadSemesters() {
    const majorSelect = document.getElementById('major_id');
    const semesterSelect = document.getElementById('semester_id');
    const submitButton = document.getElementById('submitBtn');
    
    if (!majorSelect || !semesterSelect || !submitButton) {
        console.error('Required elements not found in loadSemesters');
        return;
    }
    
    const majorId = majorSelect.value;
    const currentSemesterId = semesterSelect.value;
    
    // إعادة تعيين الفصول
    semesterSelect.innerHTML = '<option value="">-- اختر الفصل --</option>';
    semesterSelect.disabled = !majorId;
    submitButton.disabled = true;
    
    if (!majorId) {
        return;
    }
    
    // جلب الفصول عبر AJAX
    fetch(`reports.php?action=get_semesters&major_id=${majorId}`)
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok: ' + response.status);
            }
            return response.json();
        })
        .then(data => {
            if (!Array.isArray(data)) {
                console.error('Invalid data format:', data);
                alert('خطأ في البيانات المستلمة');
                return;
            }
            
            if (data.length === 0) {
                alert('لا توجد فصول دراسية متاحة لهذا التخصص');
                return;
            }
            
            data.forEach(semester => {
                const option = document.createElement('option');
                option.value = semester.id;
                option.textContent = semester.academic_year + ' - فصل ' + semester.semester_number;
                if (semester.id == currentSemesterId) {
                    option.selected = true;
                }
                semesterSelect.appendChild(option);
            });
            semesterSelect.disabled = false;
            
            // تفعيل الزر إذا كان هناك فصل محدد
            submitButton.disabled = !semesterSelect.value;
        })
        .catch(error => {
            console.error('Error loading semesters:', error);
            alert('حدث خطأ في تحميل الفصول الدراسية: ' + error.message);
        });
}

function exportToExcel(reportType) {
    let table;
    if (reportType === 'report1') {
        table = document.querySelectorAll('.report-section table')[0];
    } else if (reportType === 'report2') {
        table = document.querySelectorAll('.report-section table')[1];
    } else if (reportType === 'report3') {
        table = document.querySelectorAll('.report-section table')[2];
    } else {
        table = document.querySelector('.report-section table');
    }
    
    if (!table) {
        alert('لا توجد بيانات للتصدير');
        return;
    }
    
    let csv = '';
    const rows = table.querySelectorAll('tr');
    
    rows.forEach(row => {
        const cols = row.querySelectorAll('th, td');
        const rowData = [];
        cols.forEach(col => {
            rowData.push('"' + col.innerText.replace(/"/g, '""') + '"');
        });
        csv += rowData.join(',') + '\n';
    });
    
    // إنشاء ملف CSV وتنزيله
    const blob = new Blob(['\ufeff' + csv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    const reportNames = {
        'report1': 'تقرير_الإحصاءات',
        'report2': 'تقرير_التوزيع',
        'report3': 'تقرير_إحصائيات_الطلاب'
    };
    link.setAttribute('download', (reportNames[reportType] || 'تقرير') + '_' + new Date().getTime() + '.csv');
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// إنشاء الرسوم البيانية
document.addEventListener('DOMContentLoaded', function() {
    <?php if (!empty($report1_data)): ?>
    // الرسم البياني للتقرير الأول - المتوسطات
    const ctx1 = document.getElementById('chart1');
    if (ctx1) {
        const chart1 = new Chart(ctx1, {
            type: 'bar',
            data: {
                labels: [<?php echo implode(',', array_map(function($row) { return "'" . htmlspecialchars($row['course_name'], ENT_QUOTES) . "'"; }, $report1_data)); ?>],
                datasets: [{
                    label: 'المتوسط',
                    data: [<?php echo implode(',', array_map(function($row) { return $row['avg_grade'] !== '-' ? str_replace(',', '', $row['avg_grade']) : '0'; }, $report1_data)); ?>],
                    backgroundColor: 'rgba(102, 126, 234, 0.8)',
                    borderColor: 'rgba(102, 126, 234, 1)',
                    borderWidth: 2
                }, {
                    label: 'أصغر درجة',
                    data: [<?php echo implode(',', array_map(function($row) { return $row['min_grade'] !== '-' ? str_replace(',', '', $row['min_grade']) : '0'; }, $report1_data)); ?>],
                    backgroundColor: 'rgba(76, 175, 80, 0.8)',
                    borderColor: 'rgba(76, 175, 80, 1)',
                    borderWidth: 2
                }, {
                    label: 'أكبر درجة',
                    data: [<?php echo implode(',', array_map(function($row) { return $row['max_grade'] !== '-' ? str_replace(',', '', $row['max_grade']) : '0'; }, $report1_data)); ?>],
                    backgroundColor: 'rgba(255, 152, 0, 0.8)',
                    borderColor: 'rgba(255, 152, 0, 1)',
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'الإحصاءات الوصفية للمواد',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        display: true,
                        position: 'top'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 100
                    }
                }
            }
        });
    }
    <?php endif; ?>
    
    <?php if (!empty($report2_data)): ?>
    // الرسم البياني للتقرير الثاني - التوزيع
    const ctx2 = document.getElementById('chart2');
    if (ctx2) {
        const courseNames = [<?php echo implode(',', array_map(function($row) { return "'" . htmlspecialchars($row['course_name'], ENT_QUOTES) . "'"; }, $report2_data)); ?>];
        const chart2 = new Chart(ctx2, {
            type: 'bar',
            data: {
                labels: courseNames,
                datasets: [{
                    label: 'امتياز',
                    data: [<?php echo implode(',', array_column($report2_data, 'excellent')); ?>],
                    backgroundColor: 'rgba(76, 175, 80, 0.8)'
                }, {
                    label: 'جيد جدا',
                    data: [<?php echo implode(',', array_column($report2_data, 'very_good')); ?>],
                    backgroundColor: 'rgba(33, 150, 243, 0.8)'
                }, {
                    label: 'جيد',
                    data: [<?php echo implode(',', array_column($report2_data, 'good')); ?>],
                    backgroundColor: 'rgba(255, 193, 7, 0.8)'
                }, {
                    label: 'مقبول',
                    data: [<?php echo implode(',', array_column($report2_data, 'acceptable')); ?>],
                    backgroundColor: 'rgba(255, 152, 0, 0.8)'
                }, {
                    label: 'رسوب',
                    data: [<?php echo implode(',', array_column($report2_data, 'fail')); ?>],
                    backgroundColor: 'rgba(244, 67, 54, 0.8)'
                }, {
                    label: 'غياب',
                    data: [<?php echo implode(',', array_column($report2_data, 'absent')); ?>],
                    backgroundColor: 'rgba(158, 158, 158, 0.8)'
                }, {
                    label: 'بديل',
                    data: [<?php echo implode(',', array_column($report2_data, 'alternative')); ?>],
                    backgroundColor: 'rgba(156, 39, 176, 0.8)'
                }, {
                    label: 'غش',
                    data: [<?php echo implode(',', array_column($report2_data, 'cheating')); ?>],
                    backgroundColor: 'rgba(233, 30, 99, 0.8)'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'توزيع الدرجات حسب المادة',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        display: true,
                        position: 'top'
                    }
                },
                scales: {
                    x: {
                        stacked: false
                    },
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    }
    <?php endif; ?>
    
    <?php if (!empty($report3_data)): ?>
    // الرسم البياني للتقرير الثالث - النسب
    const ctx3 = document.getElementById('chart3');
    if (ctx3) {
        const chart3 = new Chart(ctx3, {
            type: 'doughnut',
            data: {
                labels: ['ممتاز', 'جيد جدا', 'جيد', 'مقبول', 'امتحانات البدائل', 'امتحانات الملاحق', 'العائدين للفصل', 'طلاب الغش', 'طلاب التجميد'],
                datasets: [{
                    data: [
                        <?php echo $report3_data['excellent']; ?>,
                        <?php echo $report3_data['very_good']; ?>,
                        <?php echo $report3_data['good']; ?>,
                        <?php echo $report3_data['acceptable']; ?>,
                        <?php echo $report3_data['alternative']; ?>,
                        <?php echo $report3_data['supplementary']; ?>,
                        <?php echo $report3_data['returning']; ?>,
                        <?php echo $report3_data['cheating']; ?>,
                        <?php echo $report3_data['frozen']; ?>
                    ],
                    backgroundColor: [
                        'rgba(76, 175, 80, 0.8)',
                        'rgba(33, 150, 243, 0.8)',
                        'rgba(255, 193, 7, 0.8)',
                        'rgba(255, 152, 0, 0.8)',
                        'rgba(156, 39, 176, 0.8)',
                        'rgba(244, 67, 54, 0.8)',
                        'rgba(255, 87, 34, 0.8)',
                        'rgba(233, 30, 99, 0.8)',
                        'rgba(158, 158, 158, 0.8)'
                    ],
                    borderColor: [
                        'rgba(76, 175, 80, 1)',
                        'rgba(33, 150, 243, 1)',
                        'rgba(255, 193, 7, 1)',
                        'rgba(255, 152, 0, 1)',
                        'rgba(156, 39, 176, 1)',
                        'rgba(244, 67, 54, 1)',
                        'rgba(255, 87, 34, 1)',
                        'rgba(233, 30, 99, 1)',
                        'rgba(158, 158, 158, 1)'
                    ],
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'إحصائيات الطلاب حسب التقديرات والحالات',
                        font: {
                            size: 16,
                            weight: 'bold'
                        }
                    },
                    legend: {
                        display: true,
                        position: 'right'
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                let label = context.label || '';
                                if (label) {
                                    label += ': ';
                                }
                                const value = context.parsed || 0;
                                const total = <?php echo $report3_data['total_students']; ?>;
                                const percent = total > 0 ? ((value / total) * 100).toFixed(2) : 0;
                                label += value + ' (' + percent + '%)';
                                return label;
                            }
                        }
                    }
                }
            }
        });
    }
    <?php endif; ?>
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
</body>
</html>

