<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasPermission('manage_students') && !$auth->hasRole('admin'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$student_id = isset($_GET['student_id']) ? (int)$_GET['student_id'] : 0;
$semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;
$message = '';

if (!$student_id || !$semester_id) {
    header('Location: students.php');
    exit;
}

// التحقق من الصلاحية
$student = $db->fetchOne("SELECT * FROM students WHERE id = ?", [$student_id]);
$semester = $db->fetchOne("SELECT * FROM semesters WHERE id = ?", [$semester_id]);

if (!$student || !$semester || !$auth->canAccessStudent($student_id)) {
    header('Location: students.php');
    exit;
}

// معالجة التسجيل
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'register') {
    $course_ids = $_POST['courses'] ?? [];
    $success_count = 0;
    $already_registered_count = 0;
    $error_count = 0;
    
    if (empty($course_ids)) {
        $message = '<div class="alert alert-warning">⚠️ يرجى اختيار مادة واحدة على الأقل للتسجيل</div>';
    } else {
        foreach ($course_ids as $course_id) {
            $course_id = (int)$course_id;
            
            if ($course_id <= 0) {
                continue;
            }
            
            // التحقق من التسجيل المسبق
            $existing = $db->fetchOne(
                "SELECT * FROM student_courses 
                 WHERE student_id = ? AND course_id = ? AND semester_id = ?",
                [$student_id, $course_id, $semester_id]
            );
            
            if ($existing) {
                $already_registered_count++;
                continue;
            }
            
            // محاولة التسجيل
            try {
                $db->query(
                    "INSERT INTO student_courses (student_id, course_id, semester_id, registration_date) 
                     VALUES (?, ?, ?, CURDATE())",
                    [$student_id, $course_id, $semester_id]
                );
                $success_count++;
            } catch (PDOException $e) {
                // قد تكون المادة مسجلة مسبقاً (في حالة race condition)
                if (strpos($e->getMessage(), 'Duplicate entry') !== false || 
                    strpos($e->getMessage(), 'unique_student_course_semester') !== false) {
                    $already_registered_count++;
                } else {
                    $error_count++;
                    error_log("Error registering course: " . $e->getMessage());
                }
            }
        }
        
        // بناء رسالة النتيجة
        if ($success_count > 0) {
            $message = '<div class="alert alert-success">✓ تم تسجيل ' . $success_count . ' مادة بنجاح</div>';
        }
        if ($already_registered_count > 0) {
            $message .= '<div class="alert alert-info">⚠ ' . $already_registered_count . ' مادة كانت مسجلة مسبقاً</div>';
        }
        if ($error_count > 0) {
            $message .= '<div class="alert alert-warning">⚠ ' . $error_count . ' مادة لم يتم تسجيلها</div>';
        }
        
        // إعادة تحميل الصفحة بعد النجاح
        if ($success_count > 0) {
            header("Location: register_courses.php?student_id=" . $student_id . "&semester_id=" . $semester_id . "&msg=success");
            exit;
        }
    }
}

// عرض رسالة النجاح من URL
if (isset($_GET['msg']) && $_GET['msg'] === 'success') {
    $message = '<div class="alert alert-success">✓ تم تسجيل المواد بنجاح</div>';
}

// جلب المواد المتاحة
$courses = $db->fetchAll(
    "SELECT c.* FROM courses c 
     WHERE c.major_id = ? AND (c.semester_id = ? OR c.semester_id IS NULL)
     ORDER BY c.code",
    [$student['major_id'], $semester_id]
);

// جلب المواد المسجلة
$registered = $db->fetchAll(
    "SELECT course_id FROM student_courses 
     WHERE student_id = ? AND semester_id = ?",
    [$student_id, $semester_id]
);

$registered_ids = array_column($registered, 'course_id');
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تسجيل المواد</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>تسجيل المواد - <?php echo htmlspecialchars($student['full_name_ar']); ?></h2>
            <a href="student_details.php?id=<?php echo $student_id; ?>" class="btn btn-secondary">العودة</a>
        </div>
        
        <?php echo $message; ?>
        
        <div class="card">
            <div class="card-header">
                <h3>الفصل الدراسي: <?php echo 'فصل ' . $semester['semester_number'] . ' - ' . htmlspecialchars($semester['academic_year']); ?></h3>
            </div>
            
            <form method="POST">
                <input type="hidden" name="action" value="register">
                
                <div class="table-container">
                    <table>
                        <thead>
                            <tr>
                                <th>اختيار</th>
                                <th>رمز المادة</th>
                                <th>اسم المادة</th>
                                <th>الساعات</th>
                                <th>الحالة</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($courses as $course): ?>
                                <tr>
                                    <td>
                                        <?php if (!in_array($course['id'], $registered_ids)): ?>
                                            <input type="checkbox" name="courses[]" value="<?php echo $course['id']; ?>">
                                        <?php else: ?>
                                            <span>✓</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($course['code']); ?></td>
                                    <td><?php echo htmlspecialchars($course['name_ar']); ?></td>
                                    <td><?php echo $course['credit_hours']; ?></td>
                                    <td><?php echo in_array($course['id'], $registered_ids) ? 'مسجلة' : 'غير مسجلة'; ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <button type="submit" class="btn btn-primary" onclick="return confirm('هل تريد تسجيل المواد المحددة للطالب؟');">
                    تسجيل المواد المحددة
                </button>
            </form>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

