<?php
/**
 * لوحة التحكم الرئيسية
 * Main Dashboard
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();

if (!$auth->isLoggedIn()) {
    header('Location: index.php');
    exit;
}

$user = $auth->getCurrentUser();
$db = Database::getInstance();

// إحصائيات حسب الدور
$stats = [];

if ($auth->hasRole('admin') || $auth->hasRole('college_admin')) {
    if ($auth->hasRole('admin')) {
        $stats['students'] = $db->fetchOne("SELECT COUNT(*) as count FROM students")['count'];
    } else {
        $stats['students'] = $db->fetchOne("SELECT COUNT(*) as count FROM students WHERE college_id = ?", [$user['college_id']])['count'];
    }
    $stats['active_semesters'] = $db->fetchOne("SELECT COUNT(*) as count FROM semesters WHERE is_active = 1")['count'];
    $stats['pending_certificates'] = $db->fetchOne("SELECT COUNT(*) as count FROM certificate_requests WHERE status = 'pending'")['count'];
}

if ($auth->hasRole('student')) {
    // الحصول على معلومات الطالب
    $student = $db->fetchOne(
        "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name
         FROM students s
         JOIN majors m ON s.major_id = m.id
         JOIN colleges c ON s.college_id = c.id
         WHERE s.user_id = ?",
        [$user['id']]
    );
    
    if ($student) {
        $stats['student'] = $student;
        
        // الحصول على المعدل التراكمي
        $cgpa = $db->fetchOne(
            "SELECT * FROM cumulative_gpas WHERE student_id = ?",
            [$student['id']]
        );
        $stats['cgpa'] = $cgpa;
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>لوحة التحكم - نظام نتائج جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology</title>
    <?php include __DIR__ . '/includes/compatibility.php'; ?>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        /* تنسيقات خاصة بلوحة التحكم */
        .dashboard-hero {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 20px;
            border-radius: 15px;
            margin-bottom: 30px;
            box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
            text-align: center;
        }
        
        .dashboard-hero h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            font-weight: 700;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
        }
        
        .dashboard-hero .welcome-text {
            font-size: 1.2em;
            opacity: 0.95;
            margin-top: 10px;
        }
        
        .dashboard-hero .user-role {
            display: inline-block;
            background: rgba(255,255,255,0.2);
            padding: 8px 20px;
            border-radius: 20px;
            margin-top: 15px;
            font-size: 0.9em;
            backdrop-filter: blur(10px);
        }
        
        .dashboard-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 25px;
            margin-bottom: 40px;
        }
        
        .stat-card {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            border-top: 4px solid #667eea;
        }
        
        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 100px;
            height: 100px;
            background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%);
            border-radius: 50%;
            transform: translate(30%, -30%);
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(102, 126, 234, 0.2);
        }
        
        .stat-card h3 {
            color: #666;
            font-size: 1em;
            margin-bottom: 15px;
            font-weight: 600;
            position: relative;
            z-index: 1;
        }
        
        .stat-card .stat-number {
            font-size: 2.5em;
            font-weight: 700;
            color: #667eea;
            margin: 0;
            position: relative;
            z-index: 1;
        }
        
        .stat-card .stat-icon {
            font-size: 3em;
            position: absolute;
            bottom: 20px;
            left: 20px;
            opacity: 0.1;
        }
        
        .dashboard-menu {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 20px;
            margin-top: 30px;
        }
        
        .menu-item {
            background: white;
            padding: 25px;
            border-radius: 12px;
            text-decoration: none;
            color: #333;
            font-weight: 600;
            font-size: 1.1em;
            text-align: center;
            transition: all 0.3s ease;
            box-shadow: 0 3px 15px rgba(0,0,0,0.08);
            border: 2px solid transparent;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 80px;
            position: relative;
            overflow: hidden;
        }
        
        .menu-item::before {
            content: '';
            position: absolute;
            top: 0;
            right: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
            transition: right 0.5s;
        }
        
        .menu-item:hover::before {
            right: 100%;
        }
        
        .menu-item:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.25);
            border-color: #667eea;
            color: #667eea;
        }
        
        .menu-item:active {
            transform: translateY(-1px);
        }
        
        /* ألوان خاصة لبعض القوائم */
        .menu-item.primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .menu-item.primary:hover {
            background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
            color: white;
        }
        
        .menu-item.success {
            background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
            color: white;
        }
        
        .menu-item.success:hover {
            background: linear-gradient(135deg, #45a049 0%, #4CAF50 100%);
            color: white;
        }
        
        .menu-item.warning {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
            color: white;
        }
        
        .menu-item.warning:hover {
            background: linear-gradient(135deg, #f57c00 0%, #ff9800 100%);
            color: white;
        }
        
        .menu-item.info {
            background: linear-gradient(135deg, #2196F3 0%, #1976D2 100%);
            color: white;
        }
        
        .menu-item.info:hover {
            background: linear-gradient(135deg, #1976D2 0%, #2196F3 100%);
            color: white;
        }
        
        /* تنسيقات خاصة للطلاب */
        .student-dashboard {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            padding: 30px;
            border-radius: 15px;
            margin-bottom: 30px;
        }
        
        .cgpa-display {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px;
            border-radius: 15px;
            text-align: center;
            box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
            margin-bottom: 30px;
        }
        
        .cgpa-display .cgpa-label {
            font-size: 1.2em;
            opacity: 0.9;
            margin-bottom: 10px;
        }
        
        .cgpa-display .cgpa-value {
            font-size: 4em;
            font-weight: 700;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
        }
        
        .cgpa-display .cgpa-grade {
            font-size: 1.5em;
            margin-top: 10px;
            opacity: 0.95;
        }
        
        /* تأثيرات متحركة */
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .stat-card, .menu-item {
            animation: fadeInUp 0.6s ease-out;
        }
        
        .stat-card:nth-child(1) { animation-delay: 0.1s; }
        .stat-card:nth-child(2) { animation-delay: 0.2s; }
        .stat-card:nth-child(3) { animation-delay: 0.3s; }
        .stat-card:nth-child(4) { animation-delay: 0.4s; }
        
        /* تصميم متجاوب */
        @media (max-width: 768px) {
            .dashboard-hero h1 {
                font-size: 2em;
            }
            
            .dashboard-stats {
                grid-template-columns: 1fr;
            }
            
            .dashboard-menu {
                grid-template-columns: 1fr;
            }
            
            .cgpa-display .cgpa-value {
                font-size: 3em;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <!-- قسم الترحيب -->
        <div class="dashboard-hero">
            <h1>مرحباً، <?php echo htmlspecialchars($user['full_name']); ?></h1>
            <div class="welcome-text">نظام إدارة نتائج جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology</div>
            <div class="user-role">
                <?php 
                $role_names = [
                    'admin' => 'مدير عام',
                    'college_admin' => 'مدير كلية',
                    'registrar' => 'مسجل',
                    'instructor' => 'أستاذ',
                    'student' => 'طالب'
                ];
                echo $role_names[$user['role']] ?? $user['role'];
                ?>
            </div>
        </div>
        
        <!-- إحصائيات الطلاب -->
        <?php if ($auth->hasRole('student') && isset($stats['student'])): ?>
            <?php if ($stats['cgpa']): ?>
            <div class="cgpa-display">
                <div class="cgpa-label">المعدل التراكمي</div>
                <div class="cgpa-value"><?php echo number_format($stats['cgpa']['cgpa'], 2); ?></div>
                <?php 
                $cgpa = (float)$stats['cgpa']['cgpa'];
                $grade_text = '';
                if ($cgpa >= 3.5) $grade_text = 'ممتاز';
                elseif ($cgpa >= 3.0) $grade_text = 'جيد جداً';
                elseif ($cgpa >= 2.5) $grade_text = 'جيد';
                elseif ($cgpa >= 2.0) $grade_text = 'مقبول';
                else $grade_text = 'مقبول';
                ?>
                <div class="cgpa-grade"><?php echo $grade_text; ?></div>
            </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <!-- بطاقات الإحصائيات -->
        <div class="dashboard-stats">
            <?php if ($auth->hasRole('admin') || $auth->hasRole('college_admin')): ?>
                <div class="stat-card">
                    <span class="stat-icon">👥</span>
                    <h3>الطلاب</h3>
                    <p class="stat-number"><?php echo number_format($stats['students']); ?></p>
                </div>
                <div class="stat-card">
                    <span class="stat-icon">📚</span>
                    <h3>الفصول النشطة</h3>
                    <p class="stat-number"><?php echo number_format($stats['active_semesters']); ?></p>
                </div>
                <div class="stat-card">
                    <span class="stat-icon">📜</span>
                    <h3>طلبات الشهادات المعلقة</h3>
                    <p class="stat-number"><?php echo number_format($stats['pending_certificates']); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if ($auth->hasRole('student') && isset($stats['student'])): ?>
                <div class="stat-card">
                    <span class="stat-icon">🆔</span>
                    <h3>الرقم الجامعي</h3>
                    <p class="stat-number"><?php echo htmlspecialchars($stats['student']['student_number']); ?></p>
                </div>
                <div class="stat-card">
                    <span class="stat-icon">🎓</span>
                    <h3>التخصص</h3>
                    <p class="stat-number" style="font-size: 1.2em;"><?php echo htmlspecialchars($stats['student']['major_name']); ?></p>
                </div>
                <div class="stat-card">
                    <span class="stat-icon">🏛️</span>
                    <h3>الكلية</h3>
                    <p class="stat-number" style="font-size: 1.2em;"><?php echo htmlspecialchars($stats['student']['college_name']); ?></p>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- قائمة القوائم -->
        <div class="dashboard-menu">
            <?php if ($auth->hasRole('admin') || $auth->hasRole('college_admin')): ?>
                <a href="colleges.php" class="menu-item primary">🏛️ إدارة الكليات</a>
                <a href="majors.php" class="menu-item primary">📖 إدارة التخصصات</a>
                <a href="semesters.php" class="menu-item primary">📅 إدارة الفصول الدراسية</a>
                <a href="create_next_semester.php" class="menu-item success">🔄 إنشاء فصل جديد تلقائياً</a>
                <a href="students.php" class="menu-item success">👥 إدارة الطلاب</a>
                <a href="templates_page.php" class="menu-item" style="background: linear-gradient(135deg, #ff6b6b 0%, #ee5a6f 100%); color: white;">📥 قوالب رفع البيانات (Excel)</a>
                <a href="registration/index.php" class="menu-item success">📋 نظام التسجيل الشامل</a>
                <a href="view_guide.php?file=دليل_إضافة_الطلاب_والتسجيل.md" class="menu-item info" style="background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);">📘 دليل إضافة الطلاب والتسجيل</a>
                <a href="view_guide.php?file=دليل_رفع_الطلاب_من_الإكسل.md" class="menu-item info" style="background: linear-gradient(135deg, #ff6b6b 0%, #ee5a6f 100%); color: white;">📘 دليل رفع الطلاب من Excel</a>
                <a href="view_guide.php?file=دليل_رفع_الدرجات_من_الإكسل.md" class="menu-item info" style="background: linear-gradient(135deg, #ff6b6b 0%, #ee5a6f 100%); color: white;">📘 دليل رفع الدرجات من Excel</a>
                <a href="view_guide.php?file=دليل_إدارة_الفصول_وإضافة_المواد.md" class="menu-item info" style="background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);">📅 دليل إدارة الفصول وإضافة المواد</a>
                <a href="view_guide.php?file=الدليل_الشامل_للنظام.md" class="menu-item info" style="background: linear-gradient(135deg, #9c27b0 0%, #7b1fa2 100%);">📘 الدليل الشامل للنظام</a>
                <a href="courses.php" class="menu-item success">📚 إدارة المواد</a>
                <a href="grades.php" class="menu-item info">✏️ إدخال الدرجات</a>
                <a href="enter_grades_simple.php" class="menu-item info">⚡ إدخال الدرجات (مبسط)</a>
                <a href="edit_major_semester_grades.php" class="menu-item info">📝 تعديل درجات الطلاب حسب التخصص والفصل</a>
                <a href="upload_grades.php" class="menu-item info">📤 رفع الدرجات من Excel</a>
                <a href="upload_grades_excel.php" class="menu-item info">📥 رفع درجات الفصل من Excel/CSV</a>
                <a href="semester_results.php" class="menu-item success">📊 استخراج النتائج</a>
                <a href="annual_results.php" class="menu-item success">📅 النتائج السنوية (فصلين دراسيين)</a>
                <a href="results_archive_page.php" class="menu-item info">📦 أرشيف النتائج</a>
                <a href="publish_results.php" class="menu-item success">🌐 نشر النتائج على الموقع</a>
                <a href="view_guide.php?file=دليل_نشر_النتائج.md" class="menu-item info" style="background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);">📖 دليل نشر النتائج (الشامل)</a>
                <a href="view_guide.php?file=ملخص_نشر_النتائج.md" class="menu-item info" style="background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);">⚡ ملخص نشر النتائج (سريع)</a>
                <a href="view_guide.php?file=PUBLISH_RESULTS_GUIDE.md" class="menu-item info" style="background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);">🌍 دليل نشر النتائج (إنجليزي)</a>
                <a href="college_discussion_results.php" class="menu-item success">📋 النتيجة التي ستناقش في الكلية</a>
                <a href="طلاب_الملاحق_والبدائل_حسب_المواد.php" class="menu-item success">📝 طلاب الملاحق والبدائل حسب المواد</a>
                <a href="supplementary_results_page.php" class="menu-item warning">📊 نتائج امتحانات الملاحق</a>
                <a href="removal_exam_results_page.php" class="menu-item warning">✅ نتائج امتحان إزالة الرسوب</a>
                <?php if ($auth->hasRole('admin')): ?>
                    <a href="activity_logs.php" class="menu-item warning">📋 سجل الأنشطة</a>
                <?php endif; ?>
                <a href="cumulative_gpa_report.php" class="menu-item success">📊 تقرير المعدلات التراكمية</a>
                <a href="reports.php" class="menu-item info">📈 تقارير النتائج الإحصائية</a>
                <a href="certificate_requests.php" class="menu-item warning">📜 طلبات الشهادات</a>
                <a href="fees.php" class="menu-item">💰 إدارة الرسوم</a>
                <a href="users.php" class="menu-item primary">👤 إدارة المستخدمين</a>
                <a href="add_test_grades.php" class="menu-item warning">🧪 إدخال درجات تجريبية</a>
                <a href="add_test_semesters_it.php" class="menu-item warning">📅 إدخال فصول تجريبية (تقنية المعلومات)</a>
                <a href="add_test_data_for_certificates.php" class="menu-item warning">📋 إدخال بيانات كاملة للشهادات</a>
                <a href="manage_certificate_signatures.php" class="menu-item info">✍️ إدارة أسماء المسؤولين للشهادات</a>
                <a href="upload_university_logo.php" class="menu-item info">🖼️ رفع شعار الجامعة</a>
                <a href="check_logo.php" class="menu-item info">🔍 التحقق من شعار الجامعة</a>
                <a href="help_enter_grades.php" class="menu-item info">📚 مساعدة إدخال الدرجات</a>
                <?php if ($user['role'] === 'admin' || $user['role'] === 'super_admin'): ?>
                    <a href="backup.php" class="menu-item warning">💾 النسخ الاحتياطي</a>
                    <a href="upload_database.php" class="menu-item warning">📤 رفع قاعدة البيانات</a>
                <?php endif; ?>
            <?php endif; ?>
            
            <?php if ($auth->hasRole('registrar')): ?>
                <a href="semester_results.php" class="menu-item success">📊 استخراج النتائج</a>
                <a href="annual_results.php" class="menu-item success">📅 النتائج السنوية (فصلين دراسيين)</a>
                <a href="publish_results.php" class="menu-item success">🌐 نشر النتائج على الموقع</a>
                <a href="view_guide.php?file=دليل_نشر_النتائج.md" class="menu-item info" style="background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);">📖 دليل نشر النتائج (الشامل)</a>
                <a href="view_guide.php?file=ملخص_نشر_النتائج.md" class="menu-item info" style="background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);">⚡ ملخص نشر النتائج (سريع)</a>
                <a href="view_guide.php?file=PUBLISH_RESULTS_GUIDE.md" class="menu-item info" style="background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);">🌍 دليل نشر النتائج (إنجليزي)</a>
                <a href="college_discussion_results.php" class="menu-item success">📋 النتيجة التي ستناقش في الكلية</a>
                <a href="طلاب_الملاحق_والبدائل_حسب_المواد.php" class="menu-item success">📝 طلاب الملاحق والبدائل حسب المواد</a>
                <a href="supplementary_results_page.php" class="menu-item warning">📊 نتائج امتحانات الملاحق</a>
                <a href="removal_exam_results_page.php" class="menu-item warning">✅ نتائج امتحان إزالة الرسوب</a>
                <a href="results_archive_page.php" class="menu-item info">📦 أرشيف النتائج</a>
                <a href="cumulative_gpa_report.php" class="menu-item success">📊 تقرير المعدلات التراكمية</a>
                <a href="reports.php" class="menu-item info">📈 تقارير النتائج الإحصائية</a>
                <a href="students.php" class="menu-item success">👥 إدارة الطلاب</a>
                <a href="templates_page.php" class="menu-item" style="background: linear-gradient(135deg, #ff6b6b 0%, #ee5a6f 100%); color: white;">📥 قوالب رفع البيانات (Excel)</a>
                <a href="registration/index.php" class="menu-item success">📋 نظام التسجيل الشامل</a>
                <a href="view_guide.php?file=دليل_إضافة_الطلاب_والتسجيل.md" class="menu-item info" style="background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);">📘 دليل إضافة الطلاب والتسجيل</a>
                <a href="view_guide.php?file=دليل_إدارة_الفصول_وإضافة_المواد.md" class="menu-item info" style="background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);">📅 دليل إدارة الفصول وإضافة المواد</a>
                <a href="view_guide.php?file=الدليل_الشامل_للنظام.md" class="menu-item info" style="background: linear-gradient(135deg, #9c27b0 0%, #7b1fa2 100%);">📘 الدليل الشامل للنظام</a>
                <a href="view_guide.php?file=دليل_رفع_الطلاب_من_الإكسل.md" class="menu-item info" style="background: linear-gradient(135deg, #ff6b6b 0%, #ee5a6f 100%); color: white;">📘 دليل رفع الطلاب من Excel</a>
                <a href="view_guide.php?file=دليل_رفع_الدرجات_من_الإكسل.md" class="menu-item info" style="background: linear-gradient(135deg, #ff6b6b 0%, #ee5a6f 100%); color: white;">📘 دليل رفع الدرجات من Excel</a>
                <a href="grades.php" class="menu-item info">✏️ إدخال الدرجات</a>
                <a href="enter_grades_simple.php" class="menu-item info">⚡ إدخال الدرجات (مبسط)</a>
                <a href="edit_major_semester_grades.php" class="menu-item info">📝 تعديل درجات الطلاب حسب التخصص والفصل</a>
                <a href="upload_grades.php" class="menu-item info">📤 رفع الدرجات من Excel</a>
                <a href="upload_grades_excel.php" class="menu-item info">📥 رفع درجات الفصل من Excel/CSV</a>
                <a href="help_enter_grades.php" class="menu-item info">📚 مساعدة إدخال الدرجات</a>
                <a href="certificate_requests.php" class="menu-item warning">📜 طلبات الشهادات</a>
            <?php endif; ?>
            
            <?php if ($auth->hasRole('student')): ?>
                <a href="student_grades.php" class="menu-item success">📊 عرض الدرجات</a>
                <a href="student_certificates.php" class="menu-item primary">📜 طلب شهادة</a>
            <?php endif; ?>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

