<?php
/**
 * تقرير المعدلات التراكمية للكل
 * Cumulative GPA Report for All Students
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';

$auth = new Auth();
if (!$auth->isLoggedIn()) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();

// جلب معاملات البحث
$college_id = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;

// جلب جميع الكليات
$colleges = $db->fetchAll("SELECT * FROM colleges ORDER BY name_ar");

// جلب التخصصات حسب الكلية المختارة
$majors = [];
if ($college_id > 0) {
    $majors = $db->fetchAll(
        "SELECT * FROM majors WHERE college_id = ? ORDER BY name_ar",
        [$college_id]
    );
}

// جلب معلومات الكلية والتخصص المختارين
$selected_college = null;
$selected_major = null;

if ($college_id > 0) {
    $selected_college = $db->fetchOne("SELECT * FROM colleges WHERE id = ?", [$college_id]);
}

if ($major_id > 0) {
    $selected_major = $db->fetchOne("SELECT * FROM majors WHERE id = ?", [$major_id]);
}

// جلب الطلاب حسب التخصص
$students_query = "
    SELECT s.id, s.student_number, s.full_name_ar, s.full_name_en,
           m.name_ar as major_name, m.name_en as major_name_en,
           c.name_ar as college_name, c.name_en as college_name_en,
           cg.cgpa, cg.total_credit_hours, cg.total_points
    FROM students s
    JOIN majors m ON s.major_id = m.id
    JOIN colleges c ON s.college_id = c.id
    LEFT JOIN cumulative_gpas cg ON s.id = cg.student_id
    WHERE s.status = 'active'";

$students_params = [];

if ($major_id > 0) {
    $students_query .= " AND s.major_id = ?";
    $students_params[] = $major_id;
} elseif ($college_id > 0) {
    $students_query .= " AND s.college_id = ?";
    $students_params[] = $college_id;
}

$students_query .= " ORDER BY s.student_number";

$students = $db->fetchAll($students_query, $students_params);

// حساب التقدير النصي (المعدل من 5)
function getGradeText($gpa, $failed_credits = 0) {
    // إذا كان المعدل أقل من 1.5: فصل
    if ($gpa < 1.5) {
        return 'فصل';
    }
    // إذا كان المعدل >= 1.5 و < 2.5: إعادة السنة
    elseif ($gpa >= 1.5 && $gpa < 2.5) {
        return 'إعادة السنة';
    }
    // إذا كان المعدل >= 2.5 و < 3: مقبول
    elseif ($gpa >= 2.5 && $gpa < 3) {
        // إذا رسب في مواد: ملاحق
        if ($failed_credits > 0) {
            return 'ملاحق';
        }
        return 'مقبول';
    }
    // إذا كان المعدل >= 3 و < 3.5: جيد
    elseif ($gpa >= 3 && $gpa < 3.5) {
        // إذا رسب في مواد: ملاحق
        if ($failed_credits > 0) {
            return 'ملاحق';
        }
        return 'جيد';
    }
    // إذا كان المعدل >= 3.5 و < 4: جيد جداً
    elseif ($gpa >= 3.5 && $gpa < 4) {
        // إذا رسب في مواد: ملاحق
        if ($failed_credits > 0) {
            return 'ملاحق';
        }
        return 'جيد جداً';
    }
    // إذا كان المعدل >= 4 و <= 5: ممتاز
    elseif ($gpa >= 4 && $gpa <= 5) {
        return 'ممتاز';
    }
    // حالات أخرى
    else {
        return 'مقبول';
    }
}

// حساب المعدلات التراكمية للطلاب الذين لا يملكون معدل محسوب
foreach ($students as &$student) {
    if (!$student['cgpa'] || $student['cgpa'] == 0) {
        $cgpa_data = $gradeCalculator->calculateCumulativeGPA($student['id']);
        if ($cgpa_data) {
            $student['cgpa'] = $cgpa_data['cgpa'];
            $student['total_credit_hours'] = $cgpa_data['total_credit_hours'];
            $student['total_points'] = $cgpa_data['total_points'];
        }
    }
}
unset($student);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تقرير المعدلات التراكمية - نظام نتائج جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .report-header {
            text-align: center;
            margin: 30px 0;
            padding: 20px;
            background: #f9f9f9;
            border-radius: 10px;
        }
        
        .report-header h1 {
            color: #667eea;
            font-size: 28px;
            margin: 10px 0;
        }
        
        .report-header h2 {
            color: #333;
            font-size: 22px;
            margin: 8px 0;
        }
        
        .report-header h3 {
            color: #666;
            font-size: 18px;
            margin: 5px 0;
        }
        
        .filter-section {
            background: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .filter-section form {
            display: flex;
            gap: 15px;
            align-items: flex-end;
            flex-wrap: wrap;
        }
        
        .filter-group {
            flex: 1;
            min-width: 200px;
        }
        
        .filter-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #333;
        }
        
        .filter-group select {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 14px;
        }
        
        .students-table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            background: white;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .students-table thead {
            background: #667eea;
            color: white;
        }
        
        .students-table th {
            padding: 12px;
            text-align: center;
            font-weight: bold;
            border: 1px solid #ddd;
        }
        
        .students-table td {
            padding: 10px;
            text-align: center;
            border: 1px solid #ddd;
        }
        
        .students-table tbody tr:nth-child(even) {
            background: #f9f9f9;
        }
        
        .students-table tbody tr:hover {
            background: #f0f0f0;
        }
        
        .gpa-excellent {
            color: #4CAF50;
            font-weight: bold;
        }
        
        .gpa-very-good {
            color: #2196F3;
            font-weight: bold;
        }
        
        .gpa-good {
            color: #FF9800;
            font-weight: bold;
        }
        
        .gpa-acceptable {
            color: #9E9E9E;
            font-weight: bold;
        }
        
        .print-button {
            text-align: center;
            margin: 20px 0;
        }
        
        @media print {
            .no-print {
                display: none;
            }
            
            .report-header {
                page-break-after: avoid;
            }
            
            .students-table {
                page-break-inside: auto;
            }
            
            .students-table tr {
                page-break-inside: avoid;
                page-break-after: auto;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="no-print print-button">
            <button onclick="window.print()" class="btn btn-primary">🖨️ طباعة التقرير</button>
            <a href="dashboard.php" class="btn btn-secondary">العودة</a>
        </div>
        
        <!-- قسم التصفية -->
        <div class="filter-section no-print">
            <form method="GET" action="">
                <div class="filter-group">
                    <label for="college_id">الكلية:</label>
                    <select name="college_id" id="college_id" onchange="this.form.submit()">
                        <option value="">جميع الكليات</option>
                        <?php foreach ($colleges as $college): ?>
                            <option value="<?php echo $college['id']; ?>" <?php echo $college_id == $college['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($college['name_ar']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <?php if ($college_id > 0): ?>
                <div class="filter-group">
                    <label for="major_id">التخصص:</label>
                    <select name="major_id" id="major_id" onchange="this.form.submit()">
                        <option value="">جميع التخصصات</option>
                        <?php foreach ($majors as $major): ?>
                            <option value="<?php echo $major['id']; ?>" <?php echo $major_id == $major['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($major['name_ar']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <div class="filter-group">
                    <button type="submit" class="btn btn-primary">بحث</button>
                    <a href="cumulative_gpa_report.php" class="btn btn-secondary">إعادة تعيين</a>
                </div>
            </form>
        </div>
        
        <!-- رأس التقرير -->
        <div class="report-header">
            <h1 style="font-size: 18px;">جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology</h1>
            <?php if ($selected_college): ?>
                <h2><?php echo htmlspecialchars($selected_college['name_ar']); ?></h2>
            <?php endif; ?>
            <?php if ($selected_major): ?>
                <h3><?php echo htmlspecialchars($selected_major['name_ar']); ?></h3>
            <?php endif; ?>
        </div>
        
        <!-- جدول الطلاب -->
        <?php if (!empty($students)): ?>
            <table class="students-table">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>الرقم الجامعي</th>
                        <th>اسم الطالب</th>
                        <th>مجموع النقاط</th>
                        <th>مجموع الساعات المعتمدة</th>
                        <th>المعدل التراكمي</th>
                        <th>التقدير</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $serial = 1;
                    foreach ($students as $student): 
                        $cgpa = $student['cgpa'] ?? 0;
                        $grade_text = $cgpa > 0 ? getGradeText($cgpa) : '-';
                        
                        // تحديد لون المعدل
                        $gpa_class = '';
                        if ($cgpa >= 3.5) {
                            $gpa_class = 'gpa-excellent';
                        } elseif ($cgpa >= 3.0) {
                            $gpa_class = 'gpa-very-good';
                        } elseif ($cgpa >= 2.5) {
                            $gpa_class = 'gpa-good';
                        } else {
                            $gpa_class = 'gpa-acceptable';
                        }
                    ?>
                        <tr>
                            <td><?php echo $serial++; ?></td>
                            <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                            <td><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                            <td>
                                <?php echo isset($student['total_points']) && $student['total_points'] > 0 ? number_format($student['total_points'], 2) : '-'; ?>
                            </td>
                            <td>
                                <?php echo isset($student['total_credit_hours']) && $student['total_credit_hours'] > 0 ? number_format($student['total_credit_hours'], 2) : '-'; ?>
                            </td>
                            <td class="<?php echo $gpa_class; ?>">
                                <?php echo $cgpa > 0 ? number_format($cgpa, 2) : '-'; ?>
                            </td>
                            <td class="<?php echo $gpa_class; ?>">
                                <?php echo $grade_text; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <div style="margin-top: 20px; text-align: center; color: #666;">
                <strong>إجمالي عدد الطلاب: <?php echo count($students); ?></strong>
            </div>
        <?php else: ?>
            <div class="alert alert-info" style="text-align: center; padding: 20px; margin: 20px 0;">
                لا توجد بيانات للعرض. يرجى اختيار كلية أو تخصص.
            </div>
        <?php endif; ?>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

