<?php
/**
 * إنشاء فصل دراسي جديد تلقائياً من الطلاب الناجحين والعائدين
 * Create Next Semester Automatically from Passed and Returning Students
 */
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$message = '';
$message_type = '';
$existing_semester_warning = null;

// دالة لاستخراج السنة من العام الدراسي (مثال: "2024-2025" -> [2024, 2025])
function parseAcademicYear($academic_year) {
    $parts = explode('-', $academic_year);
    if (count($parts) >= 2) {
        return [
            'start' => (int)trim($parts[0]),
            'end' => (int)trim($parts[1])
        ];
    }
    return null;
}

// دالة للحصول على العام الدراسي السابق (قبل سنة واحدة = فصلين)
function getPreviousYearAcademicYear($academic_year) {
    $year = parseAcademicYear($academic_year);
    if ($year) {
        return ($year['start'] - 1) . '-' . ($year['end'] - 1);
    }
    return null;
}

// دالة للتحقق من نجاح الطالب في فصل معين
function isStudentPassed($db, $student_id, $semester_id) {
    // جلب جميع المواد والدرجات للطالب في الفصل
    $grades = $db->fetchAll(
        "SELECT g.*, c.id as course_id, c.name_ar as course_name, c.credit_hours,
         COALESCE(g.calibrated_marks, g.marks) as display_marks,
         COALESCE(g.calibrated_letter_grade, g.letter_grade) as display_letter_grade,
         COALESCE(g.calibrated_points, g.points) as display_points
         FROM grades g
         JOIN courses c ON g.course_id = c.id
         WHERE g.student_id = ? AND g.semester_id = ?
         ORDER BY c.code",
        [$student_id, $semester_id]
    );
    
    if (empty($grades)) {
        return false; // لا توجد درجات = غير ناجح
    }
    
    // جلب قائمة المواد
    $courses = $db->fetchAll(
        "SELECT c.* FROM courses c
         JOIN grades g ON c.id = g.course_id
         WHERE g.student_id = ? AND g.semester_id = ?
         GROUP BY c.id",
        [$student_id, $semester_id]
    );
    
    if (empty($courses)) {
        return false;
    }
    
    // حساب إجمالي الساعات المعتمدة للفصل
    $total_semester_credits = 0;
    foreach ($courses as $course) {
        $total_semester_credits += (float)$course['credit_hours'];
    }
    
    // تجميع الدرجات حسب المادة
    $student_grades = [];
    foreach ($grades as $grade) {
        $student_grades[$grade['course_id']] = [
            'marks' => $grade['display_marks'],
            'letter_grade' => $grade['display_letter_grade'],
            'points' => $grade['display_points']
        ];
    }
    
    // حساب الحالة بناءً على القواعد الأكاديمية
    $failed_credits = 0;
    $is_frozen = false;
    
    foreach ($courses as $course) {
        $grade = isset($student_grades[$course['id']]) ? $student_grades[$course['id']] : null;
        
        if ($grade) {
            $letter_grade = isset($grade['letter_grade']) ? trim($grade['letter_grade']) : '';
            $marks = $grade['marks'];
            
            // التحقق من "ت" (تجميد)
            $marks_str = $marks !== null ? (string)$marks : '';
            if (strpos($marks_str, 'ت') !== false || strpos($letter_grade, 'ت') !== false) {
                return false; // مجمد = غير ناجح
            }
            
            // التحقق من الرسوب
            $is_failed = false;
            
            if ($marks !== null && !is_numeric($marks)) {
                $marks_text = trim($marks);
                if ($marks_text == 'غ' || $marks_text == 'غياب' || 
                    $marks_text == 'غش' || $marks_text == 'غشش') {
                    $is_failed = true;
                }
            } elseif ($letter_grade) {
                $letter_grade_trimmed = trim($letter_grade);
                if ($letter_grade_trimmed == 'غ' || $letter_grade_trimmed == 'غياب' ||
                    $letter_grade_trimmed == 'غش' || $letter_grade_trimmed == 'غشش' ||
                    $letter_grade_trimmed == 'F' || $letter_grade_trimmed == 'راسب') {
                    $is_failed = true;
                }
            } elseif ($marks !== null && is_numeric($marks)) {
                $is_failed = ((float)$marks < 50);
            }
            
            if ($is_failed) {
                $failed_credits += (float)$course['credit_hours'];
            }
        }
    }
    
    if ($total_semester_credits > 0) {
        $two_thirds = ($total_semester_credits * 2) / 3;
        // الطالب ناجح إذا لم يرسب في أكثر من ثلثي الساعات
        return $failed_credits <= $two_thirds;
    }
    
    return false;
}

// معالجة إنشاء الفصل الدراسي الجديد
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_semester'])) {
    $previous_semester_id = (int)$_POST['previous_semester_id'];
    $new_semester_number = (int)$_POST['new_semester_number'];
    $new_academic_year = trim($_POST['new_academic_year']);
    $new_start_date = $_POST['new_start_date'] ?? null;
    $new_end_date = $_POST['new_end_date'] ?? null;
    
    if ($previous_semester_id > 0 && $new_semester_number > 0 && !empty($new_academic_year)) {
        try {
            // جلب معلومات الفصل السابق
            $previous_semester = $db->fetchOne(
                "SELECT s.*, m.id as major_id, m.college_id 
                 FROM semesters s
                 JOIN majors m ON s.major_id = m.id
                 WHERE s.id = ?",
                [$previous_semester_id]
            );
            
            if (!$previous_semester) {
                $message = 'الفصل الدراسي السابق غير موجود';
                $message_type = 'error';
            } else {
                // التحقق من الصلاحيات
                if (!$auth->belongsToCollege($previous_semester['college_id']) && !$auth->hasRole('admin')) {
                    $message = 'ليس لديك صلاحية للوصول إلى هذا الفصل';
                    $message_type = 'error';
                } else {
                    // التحقق من وجود الفصل الجديد مسبقاً
                    $existing = $db->fetchOne(
                        "SELECT * FROM semesters 
                         WHERE major_id = ? AND semester_number = ? AND academic_year = ?",
                        [$previous_semester['major_id'], $new_semester_number, $new_academic_year]
                    );
                    
                    // التحقق إذا كان المستخدم يريد إلغاء الفصل الموجود وإنشاءه مرة أخرى
                    $force_recreate = isset($_POST['force_recreate']) && $_POST['force_recreate'] == '1';
                    
                    if ($existing && !$force_recreate) {
                        // جلب معلومات الفصل الموجود
                        $existing_semester_info = $db->fetchOne(
                            "SELECT s.*, 
                             (SELECT COUNT(DISTINCT student_id) FROM student_courses WHERE semester_id = s.id) as students_count,
                             (SELECT COUNT(*) FROM grades WHERE semester_id = s.id) as grades_count
                             FROM semesters s
                             WHERE s.id = ?",
                            [$existing['id']]
                        );
                        
                        $students_count = $existing_semester_info['students_count'] ?? 0;
                        $grades_count = $existing_semester_info['grades_count'] ?? 0;
                        
                        // حفظ معلومات الفصل الموجود في متغيرات لعرضها في الواجهة
                        $existing_semester_warning = [
                            'id' => $existing['id'],
                            'semester_number' => $existing['semester_number'],
                            'academic_year' => $existing['academic_year'],
                            'students_count' => $students_count,
                            'grades_count' => $grades_count,
                            'previous_semester_id' => $previous_semester_id,
                            'new_semester_number' => $new_semester_number,
                            'new_academic_year' => $new_academic_year,
                            'new_start_date' => $new_start_date,
                            'new_end_date' => $new_end_date
                        ];
                        
                        // لا نكمل العملية هنا - سيتم عرض رسالة تحذيرية
                    } else {
                        $db->getConnection()->beginTransaction();
                        
                        try {
                            // إذا كان المستخدم يريد إعادة الإنشاء، حذف الفصل الموجود أولاً
                            if ($existing && $force_recreate) {
                                $existing_semester_id = $existing['id'];
                                
                                // حذف جميع البيانات المرتبطة
                                // حذف تسجيلات الطلاب
                                $db->query("DELETE FROM student_courses WHERE semester_id = ?", [$existing_semester_id]);
                                // حذف الدرجات
                                $db->query("DELETE FROM grades WHERE semester_id = ?", [$existing_semester_id]);
                                // حذف المعدلات الفصلية
                                $db->query("DELETE FROM semester_gpas WHERE semester_id = ?", [$existing_semester_id]);
                                // حذف النتائج المنشورة
                                $db->query("DELETE FROM published_results WHERE semester_id = ?", [$existing_semester_id]);
                                // حذف الفصل نفسه
                                $db->query("DELETE FROM semesters WHERE id = ?", [$existing_semester_id]);
                            }
                            
                            // إنشاء الفصل الدراسي الجديد
                            $db->query(
                                "INSERT INTO semesters (major_id, semester_number, academic_year, start_date, end_date, is_active)
                                 VALUES (?, ?, ?, ?, ?, 1)",
                                [$previous_semester['major_id'], $new_semester_number, $new_academic_year, $new_start_date, $new_end_date]
                            );
                            
                            $new_semester_id = $db->getConnection()->lastInsertId();
                            
                            // 1. جلب الطلاب الناجحين من الفصل السابق
                            $all_previous_students = $db->fetchAll(
                                "SELECT DISTINCT g.student_id
                                 FROM grades g
                                 WHERE g.semester_id = ?",
                                [$previous_semester_id]
                            );
                            
                            $passed_students = [];
                            foreach ($all_previous_students as $student_data) {
                                if (isStudentPassed($db, $student_data['student_id'], $previous_semester_id)) {
                                    $passed_students[] = $student_data['student_id'];
                                }
                            }
                            
                            // 2. جلب الطلاب العائدين من الفصل قبل سنة (نفس رقم الفصل في العام السابق)
                            $previous_year = getPreviousYearAcademicYear($previous_semester['academic_year']);
                            $returning_students = [];
                            
                            if ($previous_year) {
                                $returning_semester = $db->fetchOne(
                                    "SELECT * FROM semesters
                                     WHERE major_id = ? 
                                     AND semester_number = ?
                                     AND academic_year = ?",
                                    [$previous_semester['major_id'], $new_semester_number, $previous_year]
                                );
                                
                                if ($returning_semester) {
                                    // جلب جميع الطلاب الذين درسوا الفصل السابق (من قبل سنة)
                                    $all_returning = $db->fetchAll(
                                        "SELECT DISTINCT g.student_id
                                         FROM grades g
                                         WHERE g.semester_id = ?",
                                        [$returning_semester['id']]
                                    );
                                    
                                    foreach ($all_returning as $student_data) {
                                        $student_id = $student_data['student_id'];
                                        
                                        // التحقق من أن الطالب نشط
                                        $student = $db->fetchOne(
                                            "SELECT * FROM students WHERE id = ? AND status = 'active'",
                                            [$student_id]
                                        );
                                        
                                        if (!$student) {
                                            continue;
                                        }
                                        
                                        // التحقق من أن الطالب لم يدرس الفصل الجديد أو فصول لاحقة بعد
                                        $has_later_semester = $db->fetchOne(
                                            "SELECT COUNT(*) as count FROM grades g
                                             INNER JOIN semesters s ON g.semester_id = s.id
                                             WHERE g.student_id = ? 
                                             AND s.major_id = ?
                                             AND (
                                                 (s.academic_year = ? AND s.semester_number > ?)
                                                 OR s.academic_year > ?
                                             )",
                                            [
                                                $student_id, 
                                                $previous_semester['major_id'], 
                                                $new_academic_year, 
                                                $new_semester_number,
                                                $new_academic_year
                                            ]
                                        );
                                        
                                        if ($has_later_semester['count'] == 0) {
                                            $returning_students[] = $student_id;
                                        }
                                    }
                                }
                            }
                            
                            // دمج قوائم الطلاب (إزالة التكرار)
                            $all_students_to_add = array_unique(array_merge($passed_students, $returning_students));
                            
                            // 3. إضافة الطلاب إلى الفصل الجديد (تسجيلهم في جدول student_courses)
                            // أولاً، نحتاج لجلب مواد الفصل الجديد
                            // نحصل على المواد المرتبطة برقم الفصل من فصول سابقة أو المواد العامة
                            
                            // البحث عن فصل سابق بنفس رقم الفصل في نفس التخصص (لأخذ المواد منه)
                            $similar_semester = $db->fetchOne(
                                "SELECT id FROM semesters 
                                 WHERE major_id = ? AND semester_number = ? 
                                 ORDER BY academic_year DESC LIMIT 1",
                                [$previous_semester['major_id'], $new_semester_number]
                            );
                            
                            $new_courses = [];
                            
                            if ($similar_semester) {
                                // جلب المواد من فصل سابق بنفس رقم الفصل
                                $new_courses = $db->fetchAll(
                                    "SELECT DISTINCT c.* FROM courses c
                                     WHERE c.major_id = ? 
                                     AND (c.semester_id = ? OR c.semester_id IS NULL)
                                     ORDER BY c.code",
                                    [$previous_semester['major_id'], $similar_semester['id']]
                                );
                            }
                            
                            // إذا لم توجد مواد بعد، نحصل على جميع المواد العامة للتخصص
                            if (empty($new_courses)) {
                                $new_courses = $db->fetchAll(
                                    "SELECT * FROM courses
                                     WHERE major_id = ? AND semester_id IS NULL
                                     ORDER BY code",
                                    [$previous_semester['major_id']]
                                );
                            }
                            
                            // إذا لم توجد مواد على الإطلاق، نحصل على جميع مواد التخصص
                            if (empty($new_courses)) {
                                $new_courses = $db->fetchAll(
                                    "SELECT * FROM courses
                                     WHERE major_id = ?
                                     ORDER BY code",
                                    [$previous_semester['major_id']]
                                );
                            }
                            
                            
                            $added_count = 0;
                            $errors = [];
                            
                            foreach ($all_students_to_add as $student_id) {
                                // التحقق من أن الطالب نشط
                                $student = $db->fetchOne("SELECT * FROM students WHERE id = ? AND status = 'active'", [$student_id]);
                                
                                if (!$student) {
                                    continue;
                                }
                                
                                // تسجيل الطالب في مواد الفصل الجديد
                                foreach ($new_courses as $course) {
                                    // التحقق من التسجيل المسبق
                                    $existing_registration = $db->fetchOne(
                                        "SELECT * FROM student_courses
                                         WHERE student_id = ? AND course_id = ? AND semester_id = ?",
                                        [$student_id, $course['id'], $new_semester_id]
                                    );
                                    
                                    if (!$existing_registration) {
                                        try {
                                            $db->query(
                                                "INSERT INTO student_courses (student_id, course_id, semester_id, registration_date)
                                                 VALUES (?, ?, ?, CURDATE())",
                                                [$student_id, $course['id'], $new_semester_id]
                                            );
                                        } catch (Exception $e) {
                                            $errors[] = "خطأ في تسجيل الطالب {$student_id} في المادة {$course['code']}";
                                        }
                                    }
                                }
                                
                                $added_count++;
                            }
                            
                            $db->getConnection()->commit();
                            
                            $returning_count = count($returning_students);
                            $passed_count = count($passed_students);
                            
                            $message = "تم إنشاء الفصل الدراسي الجديد بنجاح!<br>";
                            $message .= "- عدد الطلاب الناجحين من الفصل السابق: {$passed_count}<br>";
                            $message .= "- عدد الطلاب العائدين: {$returning_count}<br>";
                            $message .= "- إجمالي الطلاب المضافين: {$added_count}";
                            
                            if (!empty($errors)) {
                                $message .= "<br><br>تحذيرات: " . implode('<br>', array_slice($errors, 0, 5));
                            }
                            
                            $message_type = 'success';
                            
                        } catch (Exception $e) {
                            $db->getConnection()->rollBack();
                            $message = 'حدث خطأ أثناء إنشاء الفصل: ' . $e->getMessage();
                            $message_type = 'error';
                        }
                    }
                }
            }
        } catch (Exception $e) {
            $message = 'حدث خطأ: ' . $e->getMessage();
            $message_type = 'error';
        }
    } else {
        $message = 'يرجى ملء جميع الحقول المطلوبة';
        $message_type = 'error';
    }
}

// جلب الفصول الدراسية
$user = $auth->getCurrentUser();
$college_id = $user['college_id'] ?? null;

if ($auth->hasRole('admin')) {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name, m.code as major_code, c.name_ar as college_name
         FROM semesters s
         INNER JOIN majors m ON s.major_id = m.id
         INNER JOIN colleges c ON m.college_id = c.id
         ORDER BY s.academic_year DESC, s.semester_number DESC"
    );
} else {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name, m.code as major_code, c.name_ar as college_name
         FROM semesters s
         INNER JOIN majors m ON s.major_id = m.id
         INNER JOIN colleges c ON m.college_id = c.id
         WHERE c.id = ?
         ORDER BY s.academic_year DESC, s.semester_number DESC",
        [$college_id]
    );
}

include 'includes/header.php';
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إنشاء فصل دراسي جديد</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .semester-form {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-top: 20px;
        }
        .form-group {
            margin-bottom: 20px;
        }
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: bold;
            color: #333;
        }
        .form-group input, .form-group select {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
            box-sizing: border-box;
        }
        .info-box {
            background: #e3f2fd;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
            border-right: 4px solid #2196F3;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>إنشاء فصل دراسي جديد</h1>
        
        <?php if ($message && $message !== 'warning'): ?>
            <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?>" style="margin-bottom: 20px;">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($existing_semester_warning)): ?>
            <?php $existing_warning = $existing_semester_warning; ?>
            <div class="alert alert-warning" style="margin-bottom: 20px; padding: 20px;">
                <h3 style="margin-top: 0; color: #856404;">⚠️ تحذير: الفصل الدراسي موجود بالفعل</h3>
                <p><strong>تفاصيل الفصل الموجود:</strong></p>
                <ul style="margin-right: 20px; margin-top: 10px;">
                    <li>الفصل الدراسي: <?php echo htmlspecialchars($existing_warning['semester_number']); ?></li>
                    <li>العام الدراسي: <?php echo htmlspecialchars($existing_warning['academic_year']); ?></li>
                    <li>عدد الطلاب المسجلين: <?php echo $existing_warning['students_count']; ?></li>
                    <li>عدد الدرجات المسجلة: <?php echo $existing_warning['grades_count']; ?></li>
                </ul>
                <p style="margin-top: 15px; font-weight: bold; color: #d9534f;">
                    ⚠️ تحذير: إذا قمت بإلغاء وإنشاء الفصل مرة أخرى، سيتم حذف جميع البيانات المرتبطة به (تسجيلات الطلاب، الدرجات، المعدلات).
                </p>
                <div style="margin-top: 20px; display: flex; gap: 10px; flex-wrap: wrap;">
                    <form method="POST" style="display: inline;">
                        <input type="hidden" name="previous_semester_id" value="<?php echo htmlspecialchars($existing_warning['previous_semester_id']); ?>">
                        <input type="hidden" name="new_semester_number" value="<?php echo htmlspecialchars($existing_warning['new_semester_number']); ?>">
                        <input type="hidden" name="new_academic_year" value="<?php echo htmlspecialchars($existing_warning['new_academic_year']); ?>">
                        <input type="hidden" name="new_start_date" value="<?php echo htmlspecialchars($existing_warning['new_start_date'] ?? ''); ?>">
                        <input type="hidden" name="new_end_date" value="<?php echo htmlspecialchars($existing_warning['new_end_date'] ?? ''); ?>">
                        <input type="hidden" name="force_recreate" value="1">
                        <button type="submit" name="create_semester" class="btn btn-danger" 
                                onclick="return confirm('⚠️ تحذير خطير: سيتم حذف جميع بيانات الفصل الحالي (الطلاب، الدرجات، المعدلات). هل أنت متأكد تماماً؟')">
                            نعم، ألغِ الفصل الحالي وأنشئه مرة أخرى
                        </button>
                    </form>
                    <a href="create_next_semester.php" class="btn btn-secondary">لا، إلغاء العملية</a>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="info-box">
            <h3>ℹ️ معلومات</h3>
            <p><strong>هذه الصفحة تقوم بإنشاء فصل دراسي جديد تلقائياً:</strong></p>
            <ul style="margin-right: 20px;">
                <li>إضافة الطلاب الناجحين من الفصل الدراسي السابق في نفس السنة</li>
                <li>إضافة الطلاب العائدين من نفس الفصل في السنة السابقة (قبل سنة كاملة = فصلين دراسيين)</li>
                <li>الطلاب يتم تسجيلهم تلقائياً في جميع مواد الفصل الجديد</li>
            </ul>
            <p><strong>مثال:</strong> لإنشاء الفصل 2 من عام 2024-2025:</p>
            <ul style="margin-right: 20px;">
                <li>اختر الفصل 1 من عام 2024-2025 (الفصل السابق)</li>
                <li>النظام سيضيف الناجحين من الفصل 1</li>
                <li>النظام سيضيف العائدين من الفصل 2 من عام 2023-2024</li>
            </ul>
        </div>
        
        <div class="semester-form">
            <h2>بيانات الفصل الجديد</h2>
            <form method="POST">
                <div class="form-group">
                    <label>اختر الفصل الدراسي السابق:</label>
                    <select name="previous_semester_id" id="previous_semester" required onchange="updateNextSemester()">
                        <option value="">-- اختر الفصل الدراسي السابق --</option>
                        <?php foreach ($semesters as $semester): ?>
                            <option value="<?php echo $semester['id']; ?>" 
                                    data-semester-number="<?php echo $semester['semester_number']; ?>"
                                    data-academic-year="<?php echo htmlspecialchars($semester['academic_year']); ?>"
                                    data-major-id="<?php echo $semester['major_id']; ?>">
                                <?php echo htmlspecialchars($semester['college_name']); ?> - 
                                <?php echo htmlspecialchars($semester['major_name']); ?> - 
                                الفصل <?php echo $semester['semester_number']; ?> - 
                                <?php echo htmlspecialchars($semester['academic_year']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>رقم الفصل الدراسي الجديد:</label>
                    <input type="number" name="new_semester_number" id="new_semester_number" 
                           min="1" max="8" required>
                    <small>يجب أن يكون رقم الفصل التالي (مثلاً إذا كان السابق 1، يكون الجديد 2)</small>
                </div>
                
                <div class="form-group">
                    <label>العام الدراسي الجديد:</label>
                    <input type="text" name="new_academic_year" id="new_academic_year" 
                           placeholder="مثال: 2024-2025" required>
                    <small>إذا كان الفصل السابق في نفس السنة، استخدم نفس العام. إذا كان الفصل الأول، قد تحتاج لسنة جديدة</small>
                </div>
                
                <div class="form-group">
                    <label>تاريخ البدء (اختياري):</label>
                    <input type="date" name="new_start_date">
                </div>
                
                <div class="form-group">
                    <label>تاريخ الانتهاء (اختياري):</label>
                    <input type="date" name="new_end_date">
                </div>
                
                <button type="submit" name="create_semester" class="btn btn-primary" 
                        onclick="return confirm('هل أنت متأكد من إنشاء الفصل الدراسي الجديد؟ سيتم إضافة الطلاب تلقائياً.')">
                    إنشاء الفصل الدراسي الجديد
                </button>
            </form>
        </div>
        
        <div class="card" style="margin-top: 30px;">
            <div class="card-header">
                <h3>الفصول الدراسية المتاحة</h3>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>الكلية</th>
                            <th>التخصص</th>
                            <th>الفصل</th>
                            <th>العام الدراسي</th>
                            <th>الحالة</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($semesters)): ?>
                            <tr>
                                <td colspan="5" style="text-align: center; padding: 20px;">
                                    لا توجد فصول دراسية
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($semesters as $semester): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($semester['college_name']); ?></td>
                                    <td><?php echo htmlspecialchars($semester['major_name']); ?></td>
                                    <td>الفصل <?php echo $semester['semester_number']; ?></td>
                                    <td><?php echo htmlspecialchars($semester['academic_year']); ?></td>
                                    <td>
                                        <?php if ($semester['is_active']): ?>
                                            <span class="status-badge status-published">نشط</span>
                                        <?php else: ?>
                                            <span class="status-badge status-not-published">غير نشط</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <script>
        function updateNextSemester() {
            const select = document.getElementById('previous_semester');
            const selectedOption = select.options[select.selectedIndex];
            
            if (selectedOption.value) {
                const semesterNumber = parseInt(selectedOption.dataset.semesterNumber);
                const academicYear = selectedOption.dataset.academicYear;
                
                // حساب رقم الفصل التالي
                const nextSemesterNumber = semesterNumber + 1;
                
                // تحديث رقم الفصل الجديد
                document.getElementById('new_semester_number').value = nextSemesterNumber;
                
                // تحديث العام الدراسي (نفس العام إذا كان الفصل 1 → 2، أو قد يحتاج لسنة جديدة)
                if (semesterNumber === 1) {
                    // الفصل 1 → 2: نفس السنة
                    document.getElementById('new_academic_year').value = academicYear;
                } else {
                    // قد يحتاج لسنة جديدة (يعتمد على النظام)
                    document.getElementById('new_academic_year').value = academicYear;
                }
            }
        }
    </script>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

