<?php
/**
 * كود نتائج امتحانات الملاحق
 * Supplementary Exam Results Calculator
 * 
 * القواعد:
 * - الطالب الذي يرسب في أكثر من مادتين من مواد الملاحق: يعيد السنة الدراسية
 * - الطالب الذي يرسب في مادتين على الأكثر: يكون لديه فرصة الجلوس لامتحان إزالة الرسوب
 */

class SupplementaryResults {
    
    private $db;
    
    public function __construct($database) {
        $this->db = $database;
    }
    
    /**
     * حساب نتيجة الملاحق للطالب
     * @param int $student_id معرف الطالب
     * @param int $semester_id معرف الفصل الدراسي
     * @return array ['status' => 'repeat_year'|'removal_exam'|'pass', 'failed_count' => int, 'failed_courses' => array, 'message' => string]
     */
    public function calculateSupplementaryResult($student_id, $semester_id) {
        // جلب جميع المواد التي جلس لها الطالب في امتحانات الملاحق
        $supplementary_grades = $this->getSupplementaryGrades($student_id, $semester_id);
        
        // حساب عدد المواد الراسبة
        $failed_courses = [];
        $failed_count = 0;
        
        foreach ($supplementary_grades as $grade) {
            if ($this->isFailedGrade($grade)) {
                $failed_count++;
                $failed_courses[] = [
                    'course_id' => $grade['course_id'],
                    'course_code' => $grade['course_code'],
                    'course_name' => $grade['course_name'],
                    'marks' => $grade['marks'],
                    'letter_grade' => $grade['letter_grade']
                ];
            }
        }
        
        // تجهيز قائمة بجميع درجات الملاحق
        $all_courses_grades = [];
        foreach ($supplementary_grades as $grade) {
            $all_courses_grades[] = [
                'course_code' => $grade['course_code'],
                'course_name' => $grade['course_name'],
                'marks' => $grade['marks'],
                'letter_grade' => $grade['letter_grade']
            ];
        }
        
        // تطبيق القواعد
        $result = [
            'failed_count' => $failed_count,
            'failed_courses' => $failed_courses,
            'total_supplementary_courses' => count($supplementary_grades),
            'all_courses_grades' => $all_courses_grades
        ];
        
        if ($failed_count > 2) {
            // يرسب في أكثر من مادتين: يعيد السنة الدراسية
            $result['status'] = 'repeat_year';
            $result['message'] = 'يعيد السنة الدراسية - راسب في ' . $failed_count . ' من مواد الملاحق';
            $result['arabic_status'] = 'إعادة السنة الدراسية';
        } elseif ($failed_count <= 2 && $failed_count > 0) {
            // يرسب في مادتين على الأكثر: يسمح له بامتحان إزالة الرسوب
            $result['status'] = 'removal_exam';
            $result['message'] = 'يسمح له بامتحان إزالة الرسوب - راسب في ' . $failed_count . ' من مواد الملاحق';
            $result['arabic_status'] = 'يسمح له بامتحان إزالة الرسوب';
        } else {
            // لم يرسب في أي مادة: ناجح
            $result['status'] = 'pass';
            $result['message'] = 'ناجح في جميع مواد الملاحق';
            $result['arabic_status'] = 'ناجح';
        }
        
        return $result;
    }
    
    /**
     * جلب درجات الملاحق للطالب
     * @param int $student_id معرف الطالب
     * @param int $semester_id معرف الفصل الدراسي
     * @return array درجات الملاحق (جميع المواد التي كانت راسبة في الفصل الأصلي)
     */
    private function getSupplementaryGrades($student_id, $semester_id) {
        // جلب جميع درجات الطالب في الفصل الدراسي
        // في هذا النظام، نعتبر أن درجات الملاحق هي المواد التي كانت راسبة في الفصل الأصلي
        // نحتاج أولاً إلى معرفة المواد التي كانت راسبة في الفصل الأصلي
        // ثم جلب درجات الملاحق لهذه المواد (سواء نجحوا أم رسبوا)
        
        // جلب جميع درجات الفصل الدراسي
        $grades = $this->db->fetchAll(
            "SELECT g.*, c.code as course_code, c.name_ar as course_name, c.credit_hours,
                    COALESCE(g.calibrated_marks, g.marks) as display_marks,
                    COALESCE(g.calibrated_letter_grade, g.letter_grade) as display_letter_grade
             FROM grades g
             JOIN courses c ON g.course_id = c.id
             WHERE g.student_id = ? AND g.semester_id = ?
             ORDER BY c.code",
            [$student_id, $semester_id]
        );
        
        // جلب جميع درجات الملاحق (المواد التي كانت راسبة في الفصل الأصلي)
        // نعتبر أن المواد الراسبة حالياً هي مواد الملاحق
        $supplementary_grades = [];
        foreach ($grades as $grade) {
            // نضيف جميع المواد الراسبة في الفصل (هذه هي مواد الملاحق)
            // سواء نجح الطالب في الملاحق أم لا، يجب أن تظهر في التقرير
            $was_failed_in_original = $this->isFailedGrade($grade);
            
            if ($was_failed_in_original) {
                $supplementary_grades[] = [
                    'course_id' => $grade['course_id'],
                    'course_code' => $grade['course_code'],
                    'course_name' => $grade['course_name'],
                    'credit_hours' => $grade['credit_hours'],
                    'marks' => $grade['display_marks'],
                    'letter_grade' => $grade['display_letter_grade'],
                    'original_marks' => $grade['marks'],
                    'original_letter_grade' => $grade['letter_grade'],
                    'is_currently_failed' => $was_failed_in_original
                ];
            }
        }
        
        return $supplementary_grades;
    }
    
    /**
     * التحقق من كون الدرجة راسبة
     * @param array $grade معلومات الدرجة
     * @return bool true إذا كانت راسبة
     */
    private function isFailedGrade($grade) {
        $marks = isset($grade['display_marks']) ? $grade['display_marks'] : (isset($grade['marks']) ? $grade['marks'] : null);
        $letter_grade = isset($grade['display_letter_grade']) ? $grade['display_letter_grade'] : (isset($grade['letter_grade']) ? $grade['letter_grade'] : null);
        
        // إذا كانت الدرجة رقمية وأقل من 50
        if ($marks !== null && is_numeric($marks)) {
            return (float)$marks < 50;
        }
        
        // التحقق من القيم النصية
        if ($marks !== null && !is_numeric($marks)) {
            $marks_text = trim($marks);
            if ($marks_text == 'ب') $marks_text = 'بد';
            
            // "بد" (بديل) - لا تُحسب كراسبة
            if ($marks_text == 'بد') {
                return false;
            }
            
            // "غ" (غياب)، "غش" (غش) - تُحسب كراسبة
            if ($marks_text == 'غ' || $marks_text == 'غياب' || 
                $marks_text == 'غش' || $marks_text == 'غشش') {
                return true;
            }
            
            // "مع" (معفى)، "ت" (تجميد) - لا تُحسب كراسبة
            if ($marks_text == 'مع' || strpos($marks_text, 'مع') !== false ||
                strpos($marks_text, 'ت') !== false) {
                return false;
            }
        }
        
        // التحقق من letter_grade
        if ($letter_grade) {
            $letter_grade_trimmed = trim($letter_grade);
            if ($letter_grade_trimmed == 'ب') $letter_grade_trimmed = 'بد';
            
            if ($letter_grade_trimmed == 'بد') {
                return false;
            }
            
            if ($letter_grade_trimmed == 'غ' || $letter_grade_trimmed == 'غياب' ||
                $letter_grade_trimmed == 'غش' || $letter_grade_trimmed == 'غشش' ||
                $letter_grade_trimmed == 'F' || $letter_grade_trimmed == 'راسب') {
                return true;
            }
            
            if ($letter_grade_trimmed == 'مع' || strpos($letter_grade_trimmed, 'مع') !== false ||
                strpos($letter_grade_trimmed, 'ت') !== false) {
                return false;
            }
        }
        
        return false;
    }
    
    /**
     * حساب نتائج الملاحق لعدة طلاب
     * @param array $student_ids مصفوفة معرفات الطلاب
     * @param int $semester_id معرف الفصل الدراسي
     * @return array نتائج جميع الطلاب
     */
    public function calculateBatchSupplementaryResults($student_ids, $semester_id) {
        $results = [];
        
        foreach ($student_ids as $student_id) {
            $results[$student_id] = $this->calculateSupplementaryResult($student_id, $semester_id);
        }
        
        return $results;
    }
    
    /**
     * جلب تقرير شامل لنتائج الملاحق
     * @param int $semester_id معرف الفصل الدراسي
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $major_id معرف التخصص (اختياري)
     * @return array تقرير شامل
     */
    public function getSupplementaryReport($semester_id, $college_id = null, $major_id = null) {
        // بناء استعلام جلب الطلاب الذين لديهم مواد راسبة في الفصل (أي لديهم مواد ملاحق)
        $query = "SELECT DISTINCT s.id, s.student_number, s.full_name_ar, s.college_id, s.major_id
                  FROM students s
                  JOIN grades g ON s.id = g.student_id
                  WHERE g.semester_id = ?";
        
        $params = [$semester_id];
        
        if ($college_id) {
            $query .= " AND s.college_id = ?";
            $params[] = $college_id;
        }
        
        if ($major_id) {
            $query .= " AND s.major_id = ?";
            $params[] = $major_id;
        }
        
        $students = $this->db->fetchAll($query, $params);
        
        $report = [
            'students' => [],
            'summary' => [
                'total_students' => 0,
                'repeat_year_count' => 0,
                'removal_exam_count' => 0,
                'pass_count' => 0
            ]
        ];
        
        foreach ($students as $student) {
            $result = $this->calculateSupplementaryResult($student['id'], $semester_id);
            
            // إضافة جميع الطلاب الذين لديهم مواد ملاحق (حتى لو نجحوا جميعها)
            if ($result['total_supplementary_courses'] > 0) {
                $report['students'][] = [
                    'student_id' => $student['id'],
                    'student_number' => $student['student_number'],
                    'full_name_ar' => $student['full_name_ar'],
                    'result' => $result
                ];
                
                // تحديث الإحصائيات
                $report['summary']['total_students']++;
                if ($result['status'] == 'repeat_year') {
                    $report['summary']['repeat_year_count']++;
                } elseif ($result['status'] == 'removal_exam') {
                    $report['summary']['removal_exam_count']++;
                } elseif ($result['status'] == 'pass') {
                    $report['summary']['pass_count']++;
                }
            }
        }
        
        return $report;
    }
}

