<?php
/**
 * كود نتائج امتحان إزالة الرسوب
 * Removal Exam Results Calculator
 * 
 * القواعد:
 * - الطالب الذي يرسب في مادة واحدة في امتحان إزالة الرسوب: يعيد السنة الدراسية
 * - الطالب الذي لا يرسب في أي مادة (أو يرسب في صفر مواد): يستمر بشكل طبيعي
 */

class RemovalExamResults {
    
    private $db;
    
    public function __construct($database) {
        $this->db = $database;
    }
    
    /**
     * حساب نتيجة امتحان إزالة الرسوب للطالب
     * @param int $student_id معرف الطالب
     * @param int $semester_id معرف الفصل الدراسي
     * @return array ['status' => 'repeat_year'|'pass', 'failed_count' => int, 'failed_courses' => array, 'message' => string]
     */
    public function calculateRemovalExamResult($student_id, $semester_id) {
        // جلب جميع المواد التي جلس لها الطالب في امتحان إزالة الرسوب
        $removal_exam_grades = $this->getRemovalExamGrades($student_id, $semester_id);
        
        // حساب عدد المواد الراسبة
        $failed_courses = [];
        $failed_count = 0;
        
        foreach ($removal_exam_grades as $grade) {
            if ($this->isFailedGrade($grade)) {
                $failed_count++;
                $failed_courses[] = [
                    'course_id' => $grade['course_id'],
                    'course_code' => $grade['course_code'],
                    'course_name' => $grade['course_name'],
                    'marks' => $grade['marks'],
                    'letter_grade' => $grade['letter_grade']
                ];
            }
        }
        
        // تجهيز قائمة بجميع درجات إزالة الرسوب
        $all_courses_grades = [];
        foreach ($removal_exam_grades as $grade) {
            $all_courses_grades[] = [
                'course_code' => $grade['course_code'],
                'course_name' => $grade['course_name'],
                'marks' => $grade['marks'],
                'letter_grade' => $grade['letter_grade']
            ];
        }
        
        // تطبيق القواعد
        $result = [
            'failed_count' => $failed_count,
            'failed_courses' => $failed_courses,
            'total_removal_exam_courses' => count($removal_exam_grades),
            'all_courses_grades' => $all_courses_grades
        ];
        
        if ($failed_count == 1) {
            // يرسب في مادة واحدة: يعيد السنة الدراسية
            $result['status'] = 'repeat_year';
            $result['message'] = 'يعيد السنة الدراسية - راسب في مادة واحدة في امتحان إزالة الرسوب';
            $result['arabic_status'] = 'إعادة السنة الدراسية';
        } else {
            // لم يرسب أو رسب في صفر مواد: يستمر بشكل طبيعي
            $result['status'] = 'pass';
            if ($failed_count == 0) {
                $result['message'] = 'ناجح في جميع مواد امتحان إزالة الرسوب';
                $result['arabic_status'] = 'ناجح';
            } else {
                // حالة خاصة: إذا رسب في أكثر من مادة (لا ينبغي أن يحدث بناءً على القواعد)
                $result['message'] = 'راسب في ' . $failed_count . ' من مواد امتحان إزالة الرسوب';
                $result['arabic_status'] = 'راسب في ' . $failed_count . ' مواد';
            }
        }
        
        return $result;
    }
    
    /**
     * جلب درجات امتحان إزالة الرسوب للطالب
     * @param int $student_id معرف الطالب
     * @param int $semester_id معرف الفصل الدراسي
     * @return array درجات امتحان إزالة الرسوب (المواد التي رسب فيها في الملاحق - مادتين أو أقل)
     */
    private function getRemovalExamGrades($student_id, $semester_id) {
        // أولاً نحتاج إلى حساب نتيجة الملاحق للطالب
        // المواد المؤهلة لإزالة الرسوب هي المواد التي رسب فيها الطالب في الملاحق (مادتين أو أقل)
        
        // استخدام كلاس الملاحق للحصول على المواد الراسبة في الملاحق
        require_once __DIR__ . '/SupplementaryResults.php';
        $supplementaryResults = new SupplementaryResults($this->db);
        $supplementary_result = $supplementaryResults->calculateSupplementaryResult($student_id, $semester_id);
        
        // إذا كان الطالب مؤهل لامتحان إزالة الرسوب (رسب في مادتين أو أقل في الملاحق)
        if ($supplementary_result['status'] == 'removal_exam' || $supplementary_result['status'] == 'pass') {
            // جلب درجات الفصل الدراسي
            $grades = $this->db->fetchAll(
                "SELECT g.*, c.code as course_code, c.name_ar as course_name, c.credit_hours,
                        COALESCE(g.calibrated_marks, g.marks) as display_marks,
                        COALESCE(g.calibrated_letter_grade, g.letter_grade) as display_letter_grade
                 FROM grades g
                 JOIN courses c ON g.course_id = c.id
                 WHERE g.student_id = ? AND g.semester_id = ?
                 ORDER BY c.code",
                [$student_id, $semester_id]
            );
            
            // المواد المؤهلة لإزالة الرسوب هي المواد الراسبة في الملاحق
            // نستخدم المواد من نتيجة الملاحق
            $removal_exam_grades = [];
            
            // جلب درجات المواد التي كانت راسبة في الملاحق (حتى لو نجح الطالب فيها بعد إزالة الرسوب)
            foreach ($grades as $grade) {
                // التحقق من كون هذه المادة كانت من مواد الملاحق الراسبة
                foreach ($supplementary_result['failed_courses'] as $failed_course) {
                    if ($grade['course_id'] == $failed_course['course_id']) {
                        $removal_exam_grades[] = [
                            'course_id' => $grade['course_id'],
                            'course_code' => $grade['course_code'],
                            'course_name' => $grade['course_name'],
                            'credit_hours' => $grade['credit_hours'],
                            'marks' => $grade['display_marks'],
                            'letter_grade' => $grade['display_letter_grade'],
                            'original_marks' => $grade['marks'],
                            'original_letter_grade' => $grade['letter_grade']
                        ];
                        break;
                    }
                }
            }
            
            return $removal_exam_grades;
        }
        
        // إذا لم يكن مؤهل لإزالة الرسوب (رسب في أكثر من مادتين)، نرجع مصفوفة فارغة
        return [];
    }
    
    /**
     * التحقق من كون الدرجة راسبة
     * @param array $grade معلومات الدرجة
     * @return bool true إذا كانت راسبة
     */
    private function isFailedGrade($grade) {
        $marks = isset($grade['display_marks']) ? $grade['display_marks'] : (isset($grade['marks']) ? $grade['marks'] : null);
        $letter_grade = isset($grade['display_letter_grade']) ? $grade['display_letter_grade'] : (isset($grade['letter_grade']) ? $grade['letter_grade'] : null);
        
        // إذا كانت الدرجة رقمية وأقل من 50
        if ($marks !== null && is_numeric($marks)) {
            return (float)$marks < 50;
        }
        
        // التحقق من القيم النصية
        if ($marks !== null && !is_numeric($marks)) {
            $marks_text = trim($marks);
            if ($marks_text == 'ب') $marks_text = 'بد';
            
            // "بد" (بديل) - لا تُحسب كراسبة
            if ($marks_text == 'بد') {
                return false;
            }
            
            // "غ" (غياب)، "غش" (غش) - تُحسب كراسبة
            if ($marks_text == 'غ' || $marks_text == 'غياب' || 
                $marks_text == 'غش' || $marks_text == 'غشش') {
                return true;
            }
            
            // "مع" (معفى)، "ت" (تجميد) - لا تُحسب كراسبة
            if ($marks_text == 'مع' || strpos($marks_text, 'مع') !== false ||
                strpos($marks_text, 'ت') !== false) {
                return false;
            }
        }
        
        // التحقق من letter_grade
        if ($letter_grade) {
            $letter_grade_trimmed = trim($letter_grade);
            if ($letter_grade_trimmed == 'ب') $letter_grade_trimmed = 'بد';
            
            if ($letter_grade_trimmed == 'بد') {
                return false;
            }
            
            if ($letter_grade_trimmed == 'غ' || $letter_grade_trimmed == 'غياب' ||
                $letter_grade_trimmed == 'غش' || $letter_grade_trimmed == 'غشش' ||
                $letter_grade_trimmed == 'F' || $letter_grade_trimmed == 'راسب') {
                return true;
            }
            
            if ($letter_grade_trimmed == 'مع' || strpos($letter_grade_trimmed, 'مع') !== false ||
                strpos($letter_grade_trimmed, 'ت') !== false) {
                return false;
            }
        }
        
        return false;
    }
    
    /**
     * حساب نتائج امتحان إزالة الرسوب لعدة طلاب
     * @param array $student_ids مصفوفة معرفات الطلاب
     * @param int $semester_id معرف الفصل الدراسي
     * @return array نتائج جميع الطلاب
     */
    public function calculateBatchRemovalExamResults($student_ids, $semester_id) {
        $results = [];
        
        foreach ($student_ids as $student_id) {
            $results[$student_id] = $this->calculateRemovalExamResult($student_id, $semester_id);
        }
        
        return $results;
    }
    
    /**
     * جلب تقرير شامل لنتائج امتحان إزالة الرسوب
     * @param int $semester_id معرف الفصل الدراسي
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $major_id معرف التخصص (اختياري)
     * @return array تقرير شامل
     */
    public function getRemovalExamReport($semester_id, $college_id = null, $major_id = null) {
        // بناء استعلام جلب الطلاب
        $query = "SELECT DISTINCT s.id, s.student_number, s.full_name_ar, s.college_id, s.major_id
                  FROM students s
                  JOIN grades g ON s.id = g.student_id
                  WHERE g.semester_id = ?";
        
        $params = [$semester_id];
        
        if ($college_id) {
            $query .= " AND s.college_id = ?";
            $params[] = $college_id;
        }
        
        if ($major_id) {
            $query .= " AND s.major_id = ?";
            $params[] = $major_id;
        }
        
        $students = $this->db->fetchAll($query, $params);
        
        // استخدام كلاس الملاحق للتحقق من المؤهلين لإزالة الرسوب
        require_once __DIR__ . '/SupplementaryResults.php';
        $supplementaryResults = new SupplementaryResults($this->db);
        
        $report = [
            'students' => [],
            'summary' => [
                'total_students' => 0,
                'repeat_year_count' => 0,
                'pass_count' => 0
            ]
        ];
        
        foreach ($students as $student) {
            // التحقق من كون الطالب مؤهل لإزالة الرسوب (رسب في مادتين أو أقل في الملاحق)
            $supplementary_result = $supplementaryResults->calculateSupplementaryResult($student['id'], $semester_id);
            
            // الطالب مؤهل لإزالة الرسوب إذا كان status = 'removal_exam' أو 'pass'
            // (رسب في مادتين أو أقل، أو نجح في جميع الملاحق)
            if ($supplementary_result['status'] == 'removal_exam' || $supplementary_result['status'] == 'pass') {
                $result = $this->calculateRemovalExamResult($student['id'], $semester_id);
                
                // إضافة جميع الطلاب المؤهلين لإزالة الرسوب (حتى لو نجحوا جميع المواد)
                if ($result['total_removal_exam_courses'] > 0) {
                    $report['students'][] = [
                        'student_id' => $student['id'],
                        'student_number' => $student['student_number'],
                        'full_name_ar' => $student['full_name_ar'],
                        'result' => $result
                    ];
                    
                    // تحديث الإحصائيات
                    $report['summary']['total_students']++;
                    if ($result['status'] == 'repeat_year') {
                        $report['summary']['repeat_year_count']++;
                    } elseif ($result['status'] == 'pass') {
                        $report['summary']['pass_count']++;
                    }
                }
            }
        }
        
        return $report;
    }
}

