<?php
/**
 * قارئ ملفات Excel و CSV
 * Excel and CSV File Reader
 */

class ExcelReader {
    
    /**
     * قراءة ملف Excel أو CSV
     * @param string $file_path مسار الملف المؤقت
     * @param string $file_ext امتداد الملف (csv, xlsx, xls)
     * @return array|false بيانات الطلاب أو false في حالة الفشل
     */
    public function readFile($file_path, $file_ext) {
        if ($file_ext === 'csv') {
            return $this->readCSV($file_path);
        } elseif ($file_ext === 'xlsx' || $file_ext === 'xls') {
            return $this->readExcel($file_path, $file_ext);
        }
        return false;
    }
    
    /**
     * قراءة ملف CSV
     */
    private function readCSV($file_path) {
        $data = [];
        
        // محاولة قراءة الملف بترميز UTF-8
        $file_content = file_get_contents($file_path);
        
        // إزالة BOM إذا كان موجوداً
        if (substr($file_content, 0, 3) === "\xEF\xBB\xBF") {
            $file_content = substr($file_content, 3);
        }
        
        // التحقق من الترميز وتحويله إلى UTF-8 إذا لزم
        if (!mb_check_encoding($file_content, 'UTF-8')) {
            $file_content = mb_convert_encoding($file_content, 'UTF-8', 'auto');
        }
        
        // حفظ المحتوى المؤقت في ملف
        $temp_file = tempnam(sys_get_temp_dir(), 'csv_');
        file_put_contents($temp_file, $file_content);
        
        $handle = fopen($temp_file, 'r');
        
        if ($handle === false) {
            @unlink($temp_file);
            return false;
        }
        
        // قراءة السطر الأول (العناوين)
        $headers = fgetcsv($handle);
        if ($headers === false) {
            fclose($handle);
            @unlink($temp_file);
            return false;
        }
        
        // تنظيف العناوين وتحويل الترميز
        $original_headers = [];
        foreach ($headers as $header) {
            $header = trim($header);
            // التحقق من الترميز وتحويله
            if (!mb_check_encoding($header, 'UTF-8')) {
                $header = mb_convert_encoding($header, 'UTF-8', 'auto');
            }
            $original_headers[] = $header;
        }
        
        // قراءة البيانات - قراءة جميع الأعمدة
        $row_num = 1;
        while (($row = fgetcsv($handle)) !== false) {
            $row_num++;
            
            // تخطي الصفوف الفارغة
            if (empty(array_filter($row))) {
                continue;
            }
            
            // إنشاء مصفوفة بيانات باستخدام العناوين كمفاتيح
            $row_data = [];
            foreach ($original_headers as $index => $header) {
                $header_trimmed = trim($header);
                $header_lower = mb_strtolower($header_trimmed, 'UTF-8');
                
                // قراءة القيمة وتنظيفها
                $value = isset($row[$index]) ? trim($row[$index]) : '';
                if (!mb_check_encoding($value, 'UTF-8')) {
                    $value = mb_convert_encoding($value, 'UTF-8', 'auto');
                }
                
                // حفظ القيمة باستخدام العنوان الأصلي
                $row_data[$header] = $value;
                
                // حفظ أيضاً باستخدام العنوان الصغير للتوافق
                $row_data[$header_lower] = $value;
                
                // حفظ باستخدام الأسماء القياسية المطلوبة
                
                // student_number
                if (in_array($header_lower, ['student_number', 'student_id', 'رقم_جامعي', 'رقم جامعي', 'الرقم الجامعي', 'id', 'معرف'])) {
                    $row_data['student_number'] = $value;
                }
                
                // full_name_ar (مهم جداً!)
                if (in_array($header_lower, ['full_name_ar', 'full_name', 'name_ar', 'name', 'student_name', 'اسم', 'الاسم', 'اسم_عربي', 'اسم عربي', 'الاسم العربي', 'اسم_الطالب', 'اسم الطالب'])) {
                    $row_data['full_name_ar'] = $value;
                    $row_data['name_ar'] = $value;
                }
                
                // full_name_en
                if (in_array($header_lower, ['full_name_en', 'name_en', 'اسم_إنجليزي', 'اسم إنجليزي', 'الاسم الإنجليزي', 'english_name'])) {
                    $row_data['full_name_en'] = $value;
                    $row_data['name_en'] = $value;
                }
                
                // national_id
                if (in_array($header_lower, ['national_id', 'nid', 'رقم_وطني', 'رقم وطني', 'الرقم الوطني'])) {
                    $row_data['national_id'] = $value;
                }
                
                // email
                if (in_array($header_lower, ['email', 'e-mail', 'بريد', 'البريد الإلكتروني', 'بريد_إلكتروني'])) {
                    $row_data['email'] = $value;
                }
                
                // phone
                if (in_array($header_lower, ['phone', 'telephone', 'mobile', 'هاتف', 'الهاتف', 'رقم_الهاتف'])) {
                    $row_data['phone'] = $value;
                }
                
                // major_code
                if (in_array($header_lower, ['major_code', 'رمز_تخصص', 'رمز تخصص', 'رمز التخصص', 'code'])) {
                    $row_data['major_code'] = $value;
                }
                
                // major_id
                if (in_array($header_lower, ['major_id', 'معرف_تخصص', 'معرف تخصص'])) {
                    $row_data['major_id'] = $value;
                }
                
                // enrollment_date
                if (in_array($header_lower, ['enrollment_date', 'تاريخ_تسجيل', 'تاريخ تسجيل', 'تاريخ التسجيل', 'date', 'registration_date'])) {
                    $row_data['enrollment_date'] = $value;
                }
                
                // course_code (للدرجات)
                if (in_array($header_lower, ['course_code', 'رمز_المادة', 'رمز المادة', 'رمز', 'الرمز'])) {
                    $row_data['course_code'] = $value;
                }
                
                // course_id (للدرجات)
                if (in_array($header_lower, ['course_id', 'معرف_المادة', 'معرف المادة'])) {
                    $row_data['course_id'] = $value;
                }
                
                // marks (للدرجات)
                if (in_array($header_lower, ['marks', 'grade', 'degree', 'درجة', 'الدرجة', 'mark'])) {
                    $row_data['marks'] = $value;
                    $row_data['grade'] = $value;
                }
            }
            
            // إذا كان هناك بيانات، أضفها
            if (!empty($row_data)) {
                $data[] = $row_data;
            }
        }
        
        fclose($handle);
        @unlink($temp_file);
        return $data;
    }
    
    /**
     * قراءة ملف Excel
     */
    private function readExcel($file_path, $file_ext) {
        // محاولة استخدام مكتبة PhpSpreadsheet إذا كانت متوفرة
        if (class_exists('PhpOffice\PhpSpreadsheet\IOFactory')) {
            return $this->readExcelWithPhpSpreadsheet($file_path);
        }
        
        // إذا لم تكن متوفرة، نستخدم مكتبة بسيطة أو نعيد CSV
        // يمكن استخدام مكتبة SimpleXLSX هنا
        if (class_exists('SimpleXLSX')) {
            return $this->readExcelWithSimpleXLSX($file_path);
        }
        
        // إذا لم تكن هناك مكتبة، نحاول قراءة Excel كملف ثنائي بسيط
        // أو نطلب من المستخدم استخدام CSV
        // يمكن إضافة مكتبة بسيطة هنا أو تحويل Excel إلى CSV يدوياً
        
        // رسالة خطأ واضحة
        throw new Exception('لا توجد مكتبة لقراءة ملفات Excel. يرجى استخدام ملف CSV أو تثبيت مكتبة PhpSpreadsheet.');
    }
    
    /**
     * قراءة Excel باستخدام PhpSpreadsheet
     */
    private function readExcelWithPhpSpreadsheet($file_path) {
        try {
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file_path);
            $worksheet = $spreadsheet->getActiveSheet();
            $data = [];
            
            $highestRow = $worksheet->getHighestRow();
            $highestColumn = $worksheet->getHighestColumn();
            
            // قراءة العناوين من الصف الأول
            $original_headers = [];
            $headers_lower = [];
            $col_index = 0;
            for ($col = 'A'; $col <= $highestColumn; $col++) {
                $cellValue = $worksheet->getCell($col . '1')->getValue();
                $header = trim($cellValue);
                $original_headers[$col] = $header;
                $headers_lower[$col] = strtolower($header);
                $col_index++;
            }
            
            // قراءة البيانات من الصف الثاني
            for ($row = 2; $row <= $highestRow; $row++) {
                $row_data = [];
                
                // قراءة جميع الأعمدة
                foreach ($original_headers as $col => $header) {
                    $cellValue = $worksheet->getCell($col . $row)->getValue();
                    $value = trim($cellValue);
                    $header_lower = strtolower($header);
                    
                    // حفظ القيمة باستخدام العنوان الأصلي
                    $row_data[$header] = $value;
                    
                    // حفظ أيضاً باستخدام العنوان الصغير للتوافق
                    $row_data[$header_lower] = $value;
                    
                    // حفظ أيضاً باستخدام أسماء بديلة شائعة
                    if (in_array($header_lower, ['id', 'student_id', 'معرف', 'معرّف', 'رقم', 'الرقم'])) {
                        $row_data['student_id'] = $value;
                    }
                    if (in_array($header_lower, ['student_number', 'رقم_جامعي', 'رقم جامعي', 'الرقم الجامعي'])) {
                        $row_data['student_number'] = $value;
                    }
                    if (in_array($header_lower, ['national_id', 'nid', 'رقم_وطني', 'رقم وطني', 'الرقم الوطني'])) {
                        $row_data['national_id'] = $value;
                    }
                    if (in_array($header_lower, ['student_name', 'name', 'full_name', 'full_name_ar', 'اسم', 'الاسم', 'اسم الطالب', 'اسم_الطالب'])) {
                        $row_data['student_name'] = $value;
                    }
                    if (in_array($header_lower, ['course_code', 'course_code', 'رمز_المادة', 'رمز المادة', 'رمز', 'الرمز'])) {
                        $row_data['course_code'] = $value;
                    }
                    if (in_array($header_lower, ['course_id', 'course_id', 'معرف_المادة', 'معرف المادة'])) {
                        $row_data['course_id'] = $value;
                    }
                    if (in_array($header_lower, ['marks', 'grade', 'degree', 'درجة', 'الدرجة', 'mark', 'marks'])) {
                        $row_data['marks'] = $value;
                        $row_data['grade'] = $value;
                        $row_data['degree'] = $value;
                    }
                }
                
                // إذا كان هناك بيانات، أضفها
                if (!empty($row_data)) {
                    $data[] = $row_data;
                }
            }
            
            return $data;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * قراءة Excel باستخدام SimpleXLSX
     */
    private function readExcelWithSimpleXLSX($file_path) {
        try {
            $xlsx = SimpleXLSX::parse($file_path);
            if (!$xlsx) {
                return false;
            }
            
            $data = [];
            $rows = $xlsx->rows();
            
            if (empty($rows)) {
                return false;
            }
            
            // قراءة العناوين من الصف الأول
            $original_headers = array_map('trim', $rows[0]);
            $headers_lower = array_map('strtolower', $original_headers);
            
            // قراءة البيانات من الصف الثاني
            for ($i = 1; $i < count($rows); $i++) {
                $row = $rows[$i];
                
                // تخطي الصفوف الفارغة
                if (empty(array_filter($row))) {
                    continue;
                }
                
                // إنشاء مصفوفة بيانات باستخدام العناوين كمفاتيح
                $row_data = [];
                foreach ($original_headers as $index => $header) {
                    $header_lower = strtolower(trim($header));
                    $value = isset($row[$index]) ? trim($row[$index]) : '';
                    
                    // حفظ القيمة باستخدام العنوان الأصلي
                    $row_data[$header] = $value;
                    
                    // حفظ أيضاً باستخدام العنوان الصغير للتوافق
                    $row_data[$header_lower] = $value;
                    
                    // حفظ أيضاً باستخدام أسماء بديلة شائعة
                    if (in_array($header_lower, ['id', 'student_id', 'معرف', 'معرّف', 'رقم', 'الرقم'])) {
                        $row_data['student_id'] = $value;
                    }
                    if (in_array($header_lower, ['student_number', 'رقم_جامعي', 'رقم جامعي', 'الرقم الجامعي'])) {
                        $row_data['student_number'] = $value;
                    }
                    if (in_array($header_lower, ['national_id', 'nid', 'رقم_وطني', 'رقم وطني', 'الرقم الوطني'])) {
                        $row_data['national_id'] = $value;
                    }
                    if (in_array($header_lower, ['student_name', 'name', 'full_name', 'full_name_ar', 'اسم', 'الاسم', 'اسم الطالب', 'اسم_الطالب'])) {
                        $row_data['student_name'] = $value;
                    }
                    if (in_array($header_lower, ['course_code', 'course_code', 'رمز_المادة', 'رمز المادة', 'رمز', 'الرمز'])) {
                        $row_data['course_code'] = $value;
                    }
                    if (in_array($header_lower, ['course_id', 'course_id', 'معرف_المادة', 'معرف المادة'])) {
                        $row_data['course_id'] = $value;
                    }
                    if (in_array($header_lower, ['marks', 'grade', 'degree', 'درجة', 'الدرجة', 'mark', 'marks'])) {
                        $row_data['marks'] = $value;
                        $row_data['grade'] = $value;
                        $row_data['degree'] = $value;
                    }
                }
                
                // إذا كان هناك بيانات، أضفها
                if (!empty($row_data)) {
                    $data[] = $row_data;
                }
            }
            
            return $data;
        } catch (Exception $e) {
            return false;
        }
    }
}


