<?php
/**
 * نظام المصادقة والصلاحيات
 * Authentication and Authorization System
 */

require_once __DIR__ . '/../config/config.php';

class Auth {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * تسجيل الدخول
     */
    public function login($username, $password) {
        $user = $this->db->fetchOne(
            "SELECT u.*, c.name_ar as college_name, c.code as college_code 
             FROM users u 
             JOIN colleges c ON u.college_id = c.id 
             WHERE u.username = ? AND u.is_active = 1",
            [$username]
        );
        
        if ($user && password_verify($password, $user['password'])) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['full_name'] = $user['full_name_ar'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['college_id'] = $user['college_id'];
            $_SESSION['college_name'] = $user['college_name'];
            $_SESSION['college_code'] = $user['college_code'];
            $_SESSION['login_time'] = time();
            
            return true;
        }
        
        return false;
    }
    
    /**
     * تسجيل الخروج
     */
    public function logout() {
        // تسجيل نشاط تسجيل الخروج قبل حذف الجلسة
        if (isset($_SESSION['username'])) {
            require_once __DIR__ . '/ActivityLogger.php';
            $logger = new ActivityLogger();
            $logger->log('logout', null, null, 'تسجيل خروج المستخدم: ' . $_SESSION['username']);
        }
        
        session_unset();
        session_destroy();
    }
    
    /**
     * التحقق من تسجيل الدخول
     */
    public function isLoggedIn() {
        if (!isset($_SESSION['user_id'])) {
            return false;
        }
        
        // التحقق من انتهاء الجلسة
        if (isset($_SESSION['login_time']) && (time() - $_SESSION['login_time']) > SESSION_TIMEOUT) {
            $this->logout();
            return false;
        }
        
        // تحديث وقت الجلسة
        $_SESSION['login_time'] = time();
        
        return true;
    }
    
    /**
     * التحقق من الصلاحيات
     */
    public function hasPermission($permission) {
        if (!isset($_SESSION['role'])) {
            return false;
        }
        
        $role = $_SESSION['role'];
        
        // المدير العام لديه جميع الصلاحيات
        if ($role === 'admin') {
            return true;
        }
        
        // التحقق من الصلاحية المحددة
        $permission_check = $this->db->fetchOne(
            "SELECT * FROM permissions WHERE role = ? AND permission_name = ?",
            [$role, $permission]
        );
        
        return $permission_check !== false;
    }
    
    /**
     * التحقق من أن المستخدم ينتمي للكلية المحددة
     */
    public function belongsToCollege($college_id) {
        if (!isset($_SESSION['college_id'])) {
            return false;
        }
        
        // المدير العام يمكنه الوصول لجميع الكليات
        if ($_SESSION['role'] === 'admin') {
            return true;
        }
        
        return $_SESSION['college_id'] == $college_id;
    }
    
    /**
     * الحصول على معلومات المستخدم الحالي
     */
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['user_id'],
            'username' => $_SESSION['username'],
            'full_name' => $_SESSION['full_name'],
            'role' => $_SESSION['role'],
            'college_id' => $_SESSION['college_id'],
            'college_name' => $_SESSION['college_name']
        ];
    }
    
    /**
     * التحقق من الدور
     */
    public function hasRole($role) {
        return isset($_SESSION['role']) && $_SESSION['role'] === $role;
    }
    
    /**
     * التحقق من أن المستخدم يمكنه الوصول للطالب
     */
    public function canAccessStudent($student_id) {
        if ($this->hasRole('admin')) {
            return true;
        }
        
        $student = $this->db->fetchOne(
            "SELECT college_id FROM students WHERE id = ?",
            [$student_id]
        );
        
        return $student && $this->belongsToCollege($student['college_id']);
    }
}

