<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/CertificateGenerator.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasPermission('issue_certificates') && !$auth->hasRole('admin'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$certificateGenerator = new CertificateGenerator();
$message = '';

// معالجة الموافقة
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'approve') {
        $request_id = (int)($_POST['request_id'] ?? 0);
        $result = $certificateGenerator->approveRequest($request_id, $_SESSION['user_id']);
        $message = '<div class="alert ' . ($result['success'] ? 'alert-success' : 'alert-error') . '">' 
                   . htmlspecialchars($result['message']) . '</div>';
    } elseif ($_POST['action'] === 'reject') {
        $request_id = (int)($_POST['request_id'] ?? 0);
        $reason = $_POST['reason'] ?? '';
        
        $db->query(
            "UPDATE certificate_requests 
             SET status = 'rejected', rejection_reason = ? 
             WHERE id = ?",
            [$reason, $request_id]
        );
        $message = '<div class="alert alert-success">تم رفض الطلب</div>';
    }
}

// جلب طلبات الشهادات
$requests = [];
$debug_info = '';

try {
    // التحقق من وجود طلبات أولاً
    $total_requests = $db->fetchOne("SELECT COUNT(*) as count FROM certificate_requests");
    $total_count = $total_requests['count'] ?? 0;
    
    if ($auth->hasRole('admin')) {
        // للمدير: عرض جميع الطلبات
        $requests = $db->fetchAll(
            "SELECT cr.*, s.student_number, s.full_name_ar, c.name_ar as college_name
             FROM certificate_requests cr
             LEFT JOIN students s ON cr.student_id = s.id
             LEFT JOIN colleges c ON s.college_id = c.id
             ORDER BY cr.request_date DESC"
        );
        
        if ($total_count > 0 && empty($requests)) {
            // إذا كانت هناك طلبات لكن الاستعلام لم يعيد شيئاً، قد تكون مشكلة في JOIN
            $debug_info = "ملاحظة: يوجد {$total_count} طلب في قاعدة البيانات لكن لا يمكن عرضها. قد تكون هناك مشكلة في ارتباط الطلاب بالكليات.";
        }
    } else {
        // للمستخدمين العاديين: عرض طلبات كليتهم فقط
        if (!isset($_SESSION['college_id'])) {
            $message = '<div class="alert alert-error">خطأ: لم يتم العثور على معلومات الكلية. يرجى تسجيل الخروج والدخول مرة أخرى.</div>';
            $requests = [];
        } else {
            $college_id = $_SESSION['college_id'];
            $requests = $db->fetchAll(
                "SELECT cr.*, s.student_number, s.full_name_ar, c.name_ar as college_name
                 FROM certificate_requests cr
                 LEFT JOIN students s ON cr.student_id = s.id
                 LEFT JOIN colleges c ON s.college_id = c.id
                 WHERE s.college_id = ?
                 ORDER BY cr.request_date DESC",
                [$college_id]
            );
            
            // التحقق من الطلبات في هذه الكلية
            $college_requests_count = $db->fetchOne(
                "SELECT COUNT(*) as count 
                 FROM certificate_requests cr
                 JOIN students s ON cr.student_id = s.id
                 WHERE s.college_id = ?",
                [$college_id]
            );
            
            $college_count = $college_requests_count['count'] ?? 0;
            if ($college_count > 0 && empty($requests)) {
                $debug_info = "ملاحظة: يوجد {$college_count} طلب في كليتك لكن لا يمكن عرضها. قد تكون هناك مشكلة في البيانات.";
            }
        }
    }
    
    // إذا لم تكن هناك طلبات على الإطلاق
    if ($total_count == 0) {
        $debug_info = "لا توجد طلبات شهادات في قاعدة البيانات بعد. يمكن للطلاب تقديم طلبات من صفحة طلب الشهادات.";
    }
    
} catch (Exception $e) {
    $message = '<div class="alert alert-error">خطأ في جلب البيانات: ' . htmlspecialchars($e->getMessage()) . '</div>';
    $requests = [];
    error_log("Certificate requests error: " . $e->getMessage());
}

$certificate_types = [
    'detailed' => 'شهادة تفاصيل',
    'university' => 'شهادة جامعية',
    'transcript' => 'كشف درجات',
    'enrollment' => 'شهادة قيد',
    'graduation' => 'شهادة تخرج'
];

$status_labels = [
    'pending' => 'قيد الانتظار',
    'approved' => 'موافق عليها',
    'rejected' => 'مرفوضة',
    'issued' => 'صادرة'
];
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>طلبات الشهادات</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>طلبات الشهادات</h2>
        </div>
        
        <?php echo $message; ?>
        
        <?php if (!empty($debug_info)): ?>
            <div class="alert alert-info" style="margin: 20px 0; padding: 15px; background-color: #e3f2fd; border: 1px solid #2196f3; border-radius: 4px;">
                <strong>💡 معلومات:</strong><br>
                <?php echo htmlspecialchars($debug_info); ?>
                
                <?php if ($auth->hasRole('admin')): ?>
                    <br><br>
                    <strong>للتجربة والاختبار:</strong>
                    <ul style="margin: 10px 0; padding-right: 20px;">
                        <li>تأكد من وجود طلاب مسجلين في النظام</li>
                        <li>تأكد من أن للطلاب حسابات مستخدمين نشطة</li>
                        <li>يمكن للطلاب تقديم طلبات من صفحة <a href="student_certificates.php" style="color: #1976d2; text-decoration: underline;">طلب الشهادات</a> (إذا كانوا مسجلين دخول)</li>
                        <li>أو قم بإنشاء طلب تجريبي مباشرة من قاعدة البيانات للاختبار</li>
                    </ul>
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <?php if (empty($requests) && $auth->hasRole('admin')): ?>
            <div class="card" style="margin-bottom: 20px; background-color: #fff3cd; border: 1px solid #ffc107;">
                <div class="card-header" style="background-color: #ffc107; color: #000;">
                    <h3>📋 إرشادات لإنشاء طلبات الشهادات</h3>
                </div>
                <div style="padding: 15px;">
                    <p><strong>حالياً لا توجد طلبات شهادات في النظام.</strong></p>
                    <p>لإنشاء طلبات شهادات، يمكنك:</p>
                    <ol style="margin: 10px 0; padding-right: 25px;">
                        <li><strong>عن طريق الطلاب:</strong>
                            <ul style="margin: 5px 0; padding-right: 20px;">
                                <li>يجب أن يكون لدى الطلاب حسابات مستخدمين نشطة</li>
                                <li>يجب أن يكون الطلاب مسجلين في جدول <code>students</code></li>
                                <li>يمكن للطلاب تسجيل الدخول والذهاب إلى صفحة <strong>طلب الشهادات</strong></li>
                                <li>يمكنهم تقديم طلب شهادة من هناك</li>
                            </ul>
                        </li>
                        <li><strong>عن طريق قاعدة البيانات (للتجربة):</strong>
                            <ul style="margin: 5px 0; padding-right: 20px;">
                                <li>افتح phpMyAdmin</li>
                                <li>اذهب إلى جدول <code>certificate_requests</code></li>
                                <li>أضف سجلاً جديداً مع:
                                    <ul style="margin: 5px 0; padding-right: 15px;">
                                        <li><code>student_id</code>: رقم ID الطالب من جدول students</li>
                                        <li><code>certificate_type</code>: نوع الشهادة (detailed, university, transcript, enrollment, graduation)</li>
                                        <li><code>request_date</code>: تاريخ الطلب (مثل: 2024-10-15)</li>
                                        <li><code>status</code>: pending</li>
                                    </ul>
                                </li>
                            </ul>
                        </li>
                    </ol>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>الرقم الجامعي</th>
                        <th>اسم الطالب</th>
                        <th>الكلية</th>
                        <th>نوع الشهادة</th>
                        <th>تاريخ الطلب</th>
                        <th>الحالة</th>
                        <th>الإجراءات</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($requests)): ?>
                        <tr>
                            <td colspan="7" style="text-align: center; padding: 30px; color: #666;">
                                <div style="font-size: 16px; margin-bottom: 10px;">
                                    📭 لا توجد طلبات شهادات متاحة حالياً
                                </div>
                                <?php if (!$auth->hasRole('admin')): ?>
                                    <div style="font-size: 14px; color: #888;">
                                        سيتم عرض طلبات شهادات طلاب كليتك هنا عندما يقوم الطلاب بتقديمها
                                    </div>
                                <?php else: ?>
                                    <div style="font-size: 14px; color: #888;">
                                        سيتم عرض جميع طلبات الشهادات من جميع الكليات هنا عندما يقوم الطلاب بتقديمها
                                    </div>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($requests as $request): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($request['student_number'] ?? 'غير متاح'); ?></td>
                                <td><?php echo htmlspecialchars($request['full_name_ar'] ?? 'غير متاح'); ?></td>
                                <td><?php echo htmlspecialchars($request['college_name'] ?? 'غير متاح'); ?></td>
                                <td><?php echo $certificate_types[$request['certificate_type']] ?? $request['certificate_type'] ?? '-'; ?></td>
                                <td><?php echo !empty($request['request_date']) ? date('Y-m-d', strtotime($request['request_date'])) : '-'; ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $request['status'] ?? 'pending'; ?>">
                                        <?php echo $status_labels[$request['status']] ?? $request['status'] ?? '-'; ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if (($request['status'] ?? '') === 'pending'): ?>
                                        <form method="POST" style="display:inline;">
                                            <input type="hidden" name="action" value="approve">
                                            <input type="hidden" name="request_id" value="<?php echo $request['id']; ?>">
                                            <button type="submit" class="btn btn-success" onclick="return confirm('هل تريد الموافقة على هذا الطلب؟')">موافقة</button>
                                        </form>
                                        <button onclick="document.getElementById('reject_<?php echo $request['id']; ?>').style.display='block'" class="btn btn-danger" style="margin-top:5px;">رفض</button>
                                        
                                        <div id="reject_<?php echo $request['id']; ?>" style="display:none; margin-top:10px; padding: 15px; background-color: #f5f5f5; border-radius: 4px;">
                                            <form method="POST">
                                                <input type="hidden" name="action" value="reject">
                                                <input type="hidden" name="request_id" value="<?php echo $request['id']; ?>">
                                                <div class="form-group">
                                                    <label>سبب الرفض</label>
                                                    <textarea name="reason" required style="width: 100%; min-height: 80px; padding: 8px;"></textarea>
                                                </div>
                                                <button type="submit" class="btn btn-danger">تأكيد الرفض</button>
                                                <button type="button" onclick="document.getElementById('reject_<?php echo $request['id']; ?>').style.display='none'" class="btn btn-secondary">إلغاء</button>
                                            </form>
                                        </div>
                                    <?php elseif (($request['status'] ?? '') === 'issued'): ?>
                                        <span class="btn btn-secondary">تم الإصدار</span>
                                    <?php elseif (($request['status'] ?? '') === 'rejected'): ?>
                                        <span class="btn btn-secondary" title="<?php echo htmlspecialchars($request['rejection_reason'] ?? ''); ?>">مرفوض</span>
                                    <?php else: ?>
                                        <span class="btn btn-secondary">-</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

