<?php
/**
 * نظام النسخ الاحتياطي لقاعدة البيانات
 * Database Backup System
 * 
 * هذا الملف يقوم بعمل نسخة احتياطية كاملة من قاعدة البيانات
 * يمكن الوصول إليه من المتصفح أو من سطر الأوامر
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

// التحقق من تسجيل الدخول (فقط للمتصفح)
if (php_sapi_name() !== 'cli') {
    $auth = new Auth();
    if (!$auth->isLoggedIn()) {
        header('Location: login.php');
        exit;
    }
    
    // التحقق من الصلاحيات (فقط للمديرين)
    $user = $auth->getCurrentUser();
    if ($user['role'] !== 'admin' && $user['role'] !== 'super_admin') {
        die('ليس لديك صلاحية للوصول إلى هذه الصفحة');
    }
}

// إعدادات النسخ الاحتياطي
$backup_dir = __DIR__ . '/backups/';
$max_backups = 10; // عدد النسخ الاحتياطية المحفوظة

// إنشاء مجلد النسخ الاحتياطي إذا لم يكن موجوداً
if (!file_exists($backup_dir)) {
    @mkdir($backup_dir, 0777, true);
}

/**
 * إنشاء نسخة احتياطية باستخدام mysqldump (الطريقة المفضلة)
 */
function createBackupWithMysqldump($backup_dir) {
    $timestamp = date('Y-m-d_H-i-s');
    $filename = 'backup_' . DB_NAME . '_' . $timestamp . '.sql';
    $filepath = $backup_dir . $filename;
    
    // بناء أمر mysqldump
    $password_arg = !empty(DB_PASS) ? '-p' . escapeshellarg(DB_PASS) : '';
    $command = sprintf(
        'mysqldump -h %s -u %s %s %s > %s 2>&1',
        escapeshellarg(DB_HOST),
        escapeshellarg(DB_USER),
        $password_arg,
        escapeshellarg(DB_NAME),
        escapeshellarg($filepath)
    );
    
    // تنفيذ الأمر
    exec($command, $output, $return_var);
    
    if ($return_var === 0 && file_exists($filepath) && filesize($filepath) > 0) {
        return [
            'success' => true,
            'filename' => $filename,
            'filepath' => $filepath,
            'size' => filesize($filepath),
            'method' => 'mysqldump'
        ];
    } else {
        // حذف الملف إذا كان فارغاً أو فشل
        if (file_exists($filepath)) {
            @unlink($filepath);
        }
        return [
            'success' => false,
            'error' => 'فشل mysqldump: ' . implode("\n", $output),
            'method' => 'mysqldump'
        ];
    }
}

/**
 * إنشاء نسخة احتياطية يدوياً باستخدام PDO
 */
function createBackupManually($backup_dir) {
    $db = Database::getInstance();
    $timestamp = date('Y-m-d_H-i-s');
    $filename = 'backup_' . DB_NAME . '_' . $timestamp . '.sql';
    $filepath = $backup_dir . $filename;
    
    try {
        $fp = fopen($filepath, 'w');
        if (!$fp) {
            throw new Exception('لا يمكن إنشاء ملف النسخة الاحتياطية');
        }
        
        // كتابة رأس الملف
        fwrite($fp, "-- نسخة احتياطية من قاعدة البيانات\n");
        fwrite($fp, "-- Database Backup\n");
        fwrite($fp, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
        fwrite($fp, "-- Database: " . DB_NAME . "\n\n");
        fwrite($fp, "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n");
        fwrite($fp, "SET AUTOCOMMIT = 0;\n");
        fwrite($fp, "START TRANSACTION;\n");
        fwrite($fp, "SET time_zone = \"+00:00\";\n\n");
        
        // جلب جميع الجداول
        $tables = $db->fetchAll("SHOW TABLES");
        $table_key = 'Tables_in_' . DB_NAME;
        
        foreach ($tables as $table_row) {
            $table_name = $table_row[$table_key];
            
            // كتابة DROP TABLE
            fwrite($fp, "-- --------------------------------------------------------\n");
            fwrite($fp, "-- Table structure for table `{$table_name}`\n");
            fwrite($fp, "-- --------------------------------------------------------\n\n");
            fwrite($fp, "DROP TABLE IF EXISTS `{$table_name}`;\n");
            
            // جلب بنية الجدول
            $create_table = $db->fetchOne("SHOW CREATE TABLE `{$table_name}`");
            fwrite($fp, $create_table['Create Table'] . ";\n\n");
            
            // جلب البيانات
            $rows = $db->fetchAll("SELECT * FROM `{$table_name}`");
            
            if (count($rows) > 0) {
                fwrite($fp, "-- Dumping data for table `{$table_name}`\n\n");
                
                // جلب أسماء الأعمدة
                $columns = array_keys($rows[0]);
                $column_names = '`' . implode('`, `', $columns) . '`';
                
                foreach ($rows as $row) {
                    $values = [];
                    foreach ($row as $value) {
                        if ($value === null) {
                            $values[] = 'NULL';
                        } else {
                            // الهروب من الأحرف الخاصة
                            $escaped = addslashes($value);
                            $escaped = str_replace("\n", "\\n", $escaped);
                            $escaped = str_replace("\r", "\\r", $escaped);
                            $escaped = str_replace("\t", "\\t", $escaped);
                            $values[] = "'" . $escaped . "'";
                        }
                    }
                    
                    $values_str = implode(', ', $values);
                    fwrite($fp, "INSERT INTO `{$table_name}` ({$column_names}) VALUES ({$values_str});\n");
                }
                fwrite($fp, "\n");
            }
        }
        
        fwrite($fp, "COMMIT;\n");
        fclose($fp);
        
        return [
            'success' => true,
            'filename' => $filename,
            'filepath' => $filepath,
            'size' => filesize($filepath),
            'method' => 'manual'
        ];
        
    } catch (Exception $e) {
        if (isset($fp) && is_resource($fp)) {
            fclose($fp);
        }
        if (file_exists($filepath)) {
            @unlink($filepath);
        }
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'method' => 'manual'
        ];
    }
}

/**
 * حذف النسخ الاحتياطية القديمة
 */
function cleanupOldBackups($backup_dir, $max_backups) {
    $files = glob($backup_dir . 'backup_*.sql');
    if (count($files) > $max_backups) {
        // ترتيب الملفات حسب تاريخ التعديل (الأقدم أولاً)
        usort($files, function($a, $b) {
            return filemtime($a) - filemtime($b);
        });
        
        // حذف الملفات الزائدة
        $files_to_delete = array_slice($files, 0, count($files) - $max_backups);
        foreach ($files_to_delete as $file) {
            @unlink($file);
        }
        
        return count($files_to_delete);
    }
    return 0;
}

/**
 * تنسيق حجم الملف
 */
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// معالجة طلب إنشاء نسخة احتياطية
$backup_result = null;
$action = $_GET['action'] ?? '';

if ($action === 'create' || (php_sapi_name() === 'cli' && ($argv[1] ?? '') === 'create')) {
    // محاولة استخدام mysqldump أولاً
    $backup_result = createBackupWithMysqldump($backup_dir);
    
    // إذا فشل mysqldump، استخدم الطريقة اليدوية
    if (!$backup_result['success']) {
        $backup_result = createBackupManually($backup_dir);
    }
    
    // تنظيف النسخ القديمة
    if ($backup_result['success']) {
        cleanupOldBackups($backup_dir, $max_backups);
    }
    
    // إذا كان من سطر الأوامر، اطبع النتيجة
    if (php_sapi_name() === 'cli') {
        if ($backup_result['success']) {
            echo "✓ تم إنشاء النسخة الاحتياطية بنجاح!\n";
            echo "الملف: {$backup_result['filename']}\n";
            echo "الحجم: " . formatFileSize($backup_result['size']) . "\n";
            echo "الطريقة: {$backup_result['method']}\n";
        } else {
            echo "✗ فشل إنشاء النسخة الاحتياطية!\n";
            echo "الخطأ: {$backup_result['error']}\n";
            exit(1);
        }
        exit(0);
    }
}

// معالجة طلب تحميل نسخة احتياطية
if ($action === 'download' && isset($_GET['file'])) {
    $filename = basename($_GET['file']);
    $filepath = $backup_dir . $filename;
    
    if (file_exists($filepath) && strpos($filename, 'backup_') === 0) {
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . filesize($filepath));
        readfile($filepath);
        exit;
    } else {
        die('الملف غير موجود أو غير صالح');
    }
}

// معالجة طلب حذف نسخة احتياطية
if ($action === 'delete' && isset($_GET['file'])) {
    $filename = basename($_GET['file']);
    $filepath = $backup_dir . $filename;
    
    if (file_exists($filepath) && strpos($filename, 'backup_') === 0) {
        @unlink($filepath);
        header('Location: backup.php?deleted=1');
        exit;
    } else {
        die('الملف غير موجود أو غير صالح');
    }
}

// جلب قائمة النسخ الاحتياطية
$backup_files = glob($backup_dir . 'backup_*.sql');
usort($backup_files, function($a, $b) {
    return filemtime($b) - filemtime($a); // الأحدث أولاً
});

// إذا كان من سطر الأوامر وليس طلب إنشاء، اطبع القائمة
if (php_sapi_name() === 'cli' && ($argv[1] ?? '') !== 'create') {
    echo "═══════════════════════════════════════════════════════════\n";
    echo "   نظام النسخ الاحتياطي لقاعدة البيانات\n";
    echo "   Database Backup System\n";
    echo "═══════════════════════════════════════════════════════════\n\n";
    echo "الاستخدام:\n";
    echo "  php backup.php create    - إنشاء نسخة احتياطية جديدة\n";
    echo "  php backup.php list      - عرض قائمة النسخ الاحتياطية\n\n";
    
    if (count($backup_files) > 0) {
        echo "النسخ الاحتياطية الموجودة (" . count($backup_files) . "):\n";
        foreach ($backup_files as $file) {
            $filename = basename($file);
            $size = formatFileSize(filesize($file));
            $date = date('Y-m-d H:i:s', filemtime($file));
            echo "  - {$filename} ({$size}) - {$date}\n";
        }
    } else {
        echo "لا توجد نسخ احتياطية.\n";
    }
    exit(0);
}

// عرض واجهة HTML
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>النسخ الاحتياطي - <?php echo SITE_NAME; ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2em;
            margin-bottom: 10px;
        }
        
        .header p {
            opacity: 0.9;
        }
        
        .content {
            padding: 30px;
        }
        
        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-right: 4px solid;
        }
        
        .alert-success {
            background: #d4edda;
            border-color: #28a745;
            color: #155724;
        }
        
        .alert-error {
            background: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }
        
        .alert-info {
            background: #d1ecf1;
            border-color: #17a2b8;
            color: #0c5460;
        }
        
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            border: none;
            cursor: pointer;
            font-size: 16px;
            transition: transform 0.2s, box-shadow 0.2s;
            margin: 5px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        
        .btn-secondary {
            background: #6c757d;
        }
        
        .btn-danger {
            background: #dc3545;
        }
        
        .btn-success {
            background: #28a745;
        }
        
        .backup-list {
            margin-top: 30px;
        }
        
        .backup-item {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 15px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: box-shadow 0.2s;
        }
        
        .backup-item:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .backup-info {
            flex: 1;
        }
        
        .backup-info h3 {
            color: #333;
            margin-bottom: 8px;
        }
        
        .backup-info p {
            color: #666;
            font-size: 0.9em;
            margin: 4px 0;
        }
        
        .backup-actions {
            display: flex;
            gap: 10px;
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 10px;
            text-align: center;
        }
        
        .stat-card h3 {
            font-size: 2em;
            margin-bottom: 5px;
        }
        
        .stat-card p {
            opacity: 0.9;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        
        .empty-state svg {
            width: 100px;
            height: 100px;
            margin-bottom: 20px;
            opacity: 0.3;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>💾 النسخ الاحتياطي لقاعدة البيانات</h1>
            <p>Database Backup System</p>
        </div>
        
        <div class="content">
            <?php if (isset($_GET['deleted'])): ?>
                <div class="alert alert-success">
                    ✓ تم حذف النسخة الاحتياطية بنجاح
                </div>
            <?php endif; ?>
            
            <?php if ($backup_result): ?>
                <?php if ($backup_result['success']): ?>
                    <div class="alert alert-success">
                        <strong>✓ نجح!</strong> تم إنشاء النسخة الاحتياطية بنجاح<br>
                        <small>الملف: <?php echo htmlspecialchars($backup_result['filename']); ?> | 
                        الحجم: <?php echo formatFileSize($backup_result['size']); ?> | 
                        الطريقة: <?php echo htmlspecialchars($backup_result['method']); ?></small>
                    </div>
                <?php else: ?>
                    <div class="alert alert-error">
                        <strong>✗ فشل!</strong> <?php echo htmlspecialchars($backup_result['error']); ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
            
            <div style="text-align: center; margin-bottom: 30px;">
                <a href="backup.php?action=create" class="btn btn-success" onclick="return confirm('هل تريد إنشاء نسخة احتياطية جديدة؟')">
                    ➕ إنشاء نسخة احتياطية جديدة
                </a>
                <a href="dashboard.php" class="btn btn-secondary">🏠 العودة للوحة التحكم</a>
            </div>
            
            <div class="stats">
                <div class="stat-card">
                    <h3><?php echo count($backup_files); ?></h3>
                    <p>عدد النسخ الاحتياطية</p>
                </div>
                <div class="stat-card">
                    <h3><?php echo formatFileSize(array_sum(array_map('filesize', $backup_files))); ?></h3>
                    <p>إجمالي الحجم</p>
                </div>
                <div class="stat-card">
                    <h3><?php echo $max_backups; ?></h3>
                    <p>الحد الأقصى للنسخ</p>
                </div>
            </div>
            
            <?php if (count($backup_files) > 0): ?>
                <div class="backup-list">
                    <h2 style="margin-bottom: 20px; color: #333;">📋 قائمة النسخ الاحتياطية</h2>
                    <?php foreach ($backup_files as $file): ?>
                        <?php
                        $filename = basename($file);
                        $size = filesize($file);
                        $date = date('Y-m-d H:i:s', filemtime($file));
                        ?>
                        <div class="backup-item">
                            <div class="backup-info">
                                <h3><?php echo htmlspecialchars($filename); ?></h3>
                                <p>📅 التاريخ: <?php echo $date; ?></p>
                                <p>📦 الحجم: <?php echo formatFileSize($size); ?></p>
                            </div>
                            <div class="backup-actions">
                                <a href="backup.php?action=download&file=<?php echo urlencode($filename); ?>" class="btn">
                                    ⬇️ تحميل
                                </a>
                                <a href="backup.php?action=delete&file=<?php echo urlencode($filename); ?>" 
                                   class="btn btn-danger" 
                                   onclick="return confirm('هل أنت متأكد من حذف هذه النسخة الاحتياطية؟')">
                                    🗑️ حذف
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <svg viewBox="0 0 24 24" fill="currentColor">
                        <path d="M19 3H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm-5 14H7v-2h7v2zm3-4H7v-2h10v2zm0-4H7V7h10v2z"/>
                    </svg>
                    <h3>لا توجد نسخ احتياطية</h3>
                    <p>انقر على "إنشاء نسخة احتياطية جديدة" لإنشاء أول نسخة احتياطية</p>
                </div>
            <?php endif; ?>
            
            <div class="alert alert-info" style="margin-top: 30px;">
                <strong>💡 ملاحظات مهمة:</strong>
                <ul style="margin-right: 20px; margin-top: 10px; line-height: 1.8;">
                    <li>يتم الاحتفاظ بآخر <?php echo $max_backups; ?> نسخة احتياطية تلقائياً</li>
                    <li>يُنصح بتحميل النسخ الاحتياطية وحفظها في مكان آمن</li>
                    <li>يمكن استعادة قاعدة البيانات من خلال phpMyAdmin أو سطر الأوامر</li>
                    <li>للاستعادة من سطر الأوامر: <code>mysql -u root -p <?php echo DB_NAME; ?> < backup_file.sql</code></li>
                </ul>
            </div>
        </div>
    </div>
</body>
</html>

