<?php
/**
 * سكريبت لإدخال درجات تجريبية للاختبار
 * Script to add test grades for testing calibration
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('registrar'))) {
    die('ليس لديك صلاحية للوصول إلى هذه الصفحة');
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();
$message = '';

// معالجة إدخال الدرجات التجريبية
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_test_grades') {
    $semester_id = isset($_POST['semester_id']) ? (int)$_POST['semester_id'] : 0;
    $course_id = isset($_POST['course_id']) ? (int)$_POST['course_id'] : 0;
    
    if ($semester_id && $course_id) {
        // جلب جميع الطلاب المسجلين في المادة
        $students = $db->fetchAll(
            "SELECT sc.*, s.id as student_id 
             FROM student_courses sc
             JOIN students s ON sc.student_id = s.id
             WHERE sc.course_id = ? AND sc.semester_id = ?
             ORDER BY s.student_number",
            [$course_id, $semester_id]
        );
        
        if (empty($students)) {
            $message = '<div class="alert alert-warning">⚠️ لا يوجد طلاب مسجلين في هذه المادة</div>';
        } else {
            // إنشاء درجات تجريبية متنوعة
            $test_grades = [];
            $student_count = count($students);
            
            // توزيع الدرجات بشكل متنوع (من 50 إلى 95)
            $base_grades = [];
            if ($student_count >= 10) {
                // توزيع واسع للعدد الكبير
                $base_grades = [95, 90, 85, 80, 75, 70, 65, 60, 55, 50];
            } elseif ($student_count >= 5) {
                // توزيع متوسط
                $base_grades = [90, 80, 70, 60, 50];
            } else {
                // توزيع محدود
                $base_grades = [85, 75, 65];
            }
            
            // توزيع الدرجات على الطلاب
            for ($i = 0; $i < $student_count; $i++) {
                $index = $i % count($base_grades);
                $base_grade = $base_grades[$index];
                // إضافة تغيير بسيط عشوائي (±2)
                $grade = $base_grade + (rand(-2, 2));
                $grade = max(0, min(100, $grade)); // التأكد من أن الدرجة بين 0 و 100
                $test_grades[] = round($grade, 2);
            }
            
            // إدخال الدرجات
            $success_count = 0;
            $error_count = 0;
            
            try {
                $db->getConnection()->beginTransaction();
                
                foreach ($students as $index => $student) {
                    if (isset($test_grades[$index])) {
                        $marks = $test_grades[$index];
                        $grade_info = $gradeCalculator->calculatePoints($marks);
                        
                        // التحقق من وجود درجة سابقة
                        $existing = $db->fetchOne(
                            "SELECT * FROM grades WHERE student_course_id = ?",
                            [$student['id']]
                        );
                        
                        if ($existing) {
                            // تحديث الدرجة
                            $db->query(
                                "UPDATE grades SET marks = ?, letter_grade = ?, points = ?, updated_at = CURRENT_TIMESTAMP, entered_by = ?
                                 WHERE student_course_id = ?",
                                [$marks, $grade_info['letter'], $grade_info['points'], $_SESSION['user_id'], $student['id']]
                            );
                        } else {
                            // إضافة درجة جديدة
                            $db->query(
                                "INSERT INTO grades (student_course_id, student_id, course_id, semester_id, marks, letter_grade, points, entered_by) 
                                 VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
                                [
                                    $student['id'],
                                    $student['student_id'],
                                    $course_id,
                                    $semester_id,
                                    $marks,
                                    $grade_info['letter'],
                                    $grade_info['points'],
                                    $_SESSION['user_id']
                                ]
                            );
                        }
                        
                        // تحديث المعدلات
                        $gradeCalculator->updateAllGPAs($student['student_id']);
                        $success_count++;
                    }
                }
                
                $db->getConnection()->commit();
                $message = '<div class="alert alert-success">✅ تم إدخال ' . $success_count . ' درجة تجريبية بنجاح!</div>';
            } catch (Exception $e) {
                $db->getConnection()->rollBack();
                $message = '<div class="alert alert-error">❌ خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
        }
    }
}

// جلب الفصول الدراسية
$semesters = [];
if ($auth->hasRole('admin')) {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         ORDER BY s.academic_year DESC, s.semester_number DESC"
    );
} else {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         WHERE m.college_id = ?
         ORDER BY s.academic_year DESC, s.semester_number DESC",
        [$_SESSION['college_id']]
    );
}

// جلب المواد حسب الفصل المختار
$courses = [];
$selected_semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;
if ($selected_semester_id) {
    $semester = $db->fetchOne("SELECT * FROM semesters WHERE id = ?", [$selected_semester_id]);
    if ($semester) {
        $courses = $db->fetchAll(
            "SELECT c.* FROM courses c 
             WHERE c.major_id = ? AND (c.semester_id = ? OR c.semester_id IS NULL)
             ORDER BY c.code",
            [$semester['major_id'], $selected_semester_id]
        );
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدخال درجات تجريبية</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .info-box {
            background: #e3f2fd;
            border-right: 5px solid #2196F3;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
        }
        .warning-box {
            background: #fff3cd;
            border-right: 5px solid #ffc107;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>إدخال درجات تجريبية للاختبار</h2>
            <p>هذه الصفحة لإدخال درجات تجريبية لاختبار نظام المعايرة</p>
        </div>
        
        <?php echo $message; ?>
        
        <div class="warning-box">
            <strong>⚠️ تحذير:</strong> هذه الصفحة لإدخال درجات تجريبية فقط. 
            <ul style="margin-right: 20px; margin-top: 10px;">
                <li>سيتم إدخال درجات متنوعة تلقائياً (من 50 إلى 95)</li>
                <li>إذا كانت هناك درجات موجودة، سيتم تحديثها</li>
                <li>استخدم هذه الصفحة فقط للاختبار</li>
            </ul>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h3>الخطوة 1: اختر الفصل الدراسي</h3>
            </div>
            <form method="GET" style="display:inline;">
                <div class="form-group">
                    <label>الفصل الدراسي</label>
                    <select name="semester_id" onchange="this.form.submit()" style="width:100%; padding:10px;">
                        <option value="">-- اختر الفصل الدراسي --</option>
                        <?php foreach ($semesters as $sem): ?>
                            <option value="<?php echo $sem['id']; ?>" <?php echo $sem['id'] == $selected_semester_id ? 'selected' : ''; ?>>
                                فصل <?php echo $sem['semester_number']; ?> - <?php echo htmlspecialchars($sem['academic_year']); ?> 
                                (<?php echo htmlspecialchars($sem['major_name']); ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </form>
        </div>
        
        <?php if ($selected_semester_id): ?>
            <?php if (empty($courses)): ?>
                <div class="alert alert-info">
                    <strong>لا توجد مواد في هذا الفصل الدراسي</strong><br>
                    يجب إضافة المواد أولاً من صفحة <a href="courses.php">إدارة المواد</a>
                </div>
            <?php else: ?>
                <div class="card">
                    <div class="card-header">
                        <h3>الخطوة 2: اختر المادة</h3>
                    </div>
                    <form method="POST">
                        <input type="hidden" name="action" value="add_test_grades">
                        <input type="hidden" name="semester_id" value="<?php echo $selected_semester_id; ?>">
                        <div class="form-group">
                            <label>المادة</label>
                            <select name="course_id" required style="width:100%; padding:10px;">
                                <option value="">-- اختر المادة --</option>
                                <?php foreach ($courses as $c): ?>
                                    <?php
                                    // حساب عدد الطلاب المسجلين
                                    $student_count = $db->fetchOne(
                                        "SELECT COUNT(*) as count FROM student_courses 
                                         WHERE course_id = ? AND semester_id = ?",
                                        [$c['id'], $selected_semester_id]
                                    )['count'];
                                    
                                    // حساب عدد الدرجات الموجودة
                                    $grades_count = $db->fetchOne(
                                        "SELECT COUNT(*) as count FROM grades g
                                         JOIN student_courses sc ON g.student_course_id = sc.id
                                         WHERE sc.course_id = ? AND sc.semester_id = ?",
                                        [$c['id'], $selected_semester_id]
                                    )['count'];
                                    ?>
                                    <option value="<?php echo $c['id']; ?>">
                                        <?php echo htmlspecialchars($c['code']); ?> - <?php echo htmlspecialchars($c['name_ar']); ?> 
                                        (<?php echo $c['credit_hours']; ?> ساعات)
                                        - <?php echo $student_count; ?> طالب مسجل
                                        <?php if ($grades_count > 0): ?>
                                            (<?php echo $grades_count; ?> درجة موجودة)
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="info-box">
                            <strong>ℹ️ معلومات:</strong>
                            <ul style="margin-right: 20px; margin-top: 10px;">
                                <li>سيتم إدخال درجات متنوعة تلقائياً (من 50 إلى 95)</li>
                                <li>الدرجات ستكون متنوعة لضمان عمل المعايرة بشكل صحيح</li>
                                <li>إذا كانت هناك درجات موجودة، سيتم تحديثها</li>
                                <li>سيتم تحديث المعدلات تلقائياً</li>
                            </ul>
                        </div>
                        
                        <button type="submit" class="btn btn-primary" style="padding: 12px 30px; font-size: 16px; margin-top: 15px;">
                            ✅ إدخال درجات تجريبية
                        </button>
                    </form>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <div class="card" style="margin-top: 20px;">
            <div class="card-header">
                <h3>📋 الخطوات التالية</h3>
            </div>
            <ol style="margin-right: 20px;">
                <li>اختر الفصل الدراسي</li>
                <li>اختر المادة</li>
                <li>اضغط "إدخال درجات تجريبية"</li>
                <li>اذهب إلى <a href="semester_results.php">استخراج النتائج</a></li>
                <li>اختر الفصل الدراسي نفسه</li>
                <li>اضغط "النتائج بعد المعايرة (ثوراندويك)"</li>
                <li>ستظهر الدرجات المعايرة</li>
            </ol>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

